"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getChangePoints = getChangePoints;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function getMessage({
  changes,
  startTime,
  endTime
}) {
  switch (changes.type) {
    case 'spike':
      return `A significant spike (transient increase) was detected for the time range ${startTime} to ${endTime} at ${changes.time} (p_value: ${changes.p_value}).`;
    case 'dip':
      return `A significant dip (transient decrease) was detected for the time range ${startTime} to ${endTime} at ${changes.time} (p_value: ${changes.p_value}).`;
    case 'step_change':
      return `A step change (baseline shift) was detected for the time range ${startTime} to ${endTime} starting at ${changes.time} (p_value: ${changes.p_value}).`;
    case 'trend_change':
      return `A trend change (slope shift) was detected for the time range ${startTime} to ${endTime} starting at ${changes.time} (p_value: ${changes.p_value}).`;
    case 'stationary':
      return `No change points were found for the time range ${startTime} to ${endTime}. The data is stationary (stable).`;
    case 'non_stationary':
      return `No specific change point was found for the time range ${startTime} to ${endTime}, but the data is non-stationary (unstable). The values are drifting or volatile without a clear single trigger event.`;
    case 'indeterminable':
      return `No specific change point was found for the time range ${startTime} to ${endTime}, and the data is indeterminable. There is insufficient evidence to classify the data as stationary or non-stationary.`;
    default:
      return `An anomaly of type '${changes.type}' was detected for the time range ${startTime} to ${endTime} at ${changes.time}.`;
  }
}
function isChangePoint(changes) {
  return !changes.type.stationary && !changes.type.indeterminable && !changes.type.non_stationary;
}
async function getChangePoints({
  buckets
}) {
  const series = buckets.filter(bucket => bucket.changes).map(bucket => {
    var _changesResult$bucket, _bucket$time_series$b, _bucket$time_series$b2, _bucket$time_series$b3, _bucket$time_series$b4;
    const changesResult = bucket.changes;
    const [changeType, value] = Object.entries(changesResult.type)[0];
    const changes = {
      time: (_changesResult$bucket = changesResult.bucket) !== null && _changesResult$bucket !== void 0 && _changesResult$bucket.key ? new Date(changesResult.bucket.key).toISOString() : undefined,
      type: changeType,
      ...value
    };
    return {
      key: bucket.key,
      pattern: bucket.regex,
      message: getMessage({
        changes,
        startTime: (_bucket$time_series$b = (_bucket$time_series$b2 = bucket.time_series.buckets[0]) === null || _bucket$time_series$b2 === void 0 ? void 0 : _bucket$time_series$b2.key_as_string) !== null && _bucket$time_series$b !== void 0 ? _bucket$time_series$b : '',
        endTime: (_bucket$time_series$b3 = (_bucket$time_series$b4 = bucket.time_series.buckets[bucket.time_series.buckets.length - 1]) === null || _bucket$time_series$b4 === void 0 ? void 0 : _bucket$time_series$b4.key_as_string) !== null && _bucket$time_series$b3 !== void 0 ? _bucket$time_series$b3 : ''
      }),
      timeSeries: isChangePoint(bucket.changes) ? bucket.time_series.buckets.map(group => ({
        x: new Date(group.key).getTime(),
        y: group.doc_count
      })) : undefined,
      changes
    };
  });
  return series;
}