"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.OBSERVABILITY_RUN_LOG_RATE_ANALYSIS_TOOL_ID = void 0;
exports.createRunLogRateAnalysisTool = createRunLogRateAnalysisTool;
var _zod = require("@kbn/zod");
var _onechatCommon = require("@kbn/onechat-common");
var _tool_result = require("@kbn/onechat-common/tools/tool_result");
var _get_agent_builder_resource_availability = require("../../utils/get_agent_builder_resource_availability");
var _tool_schemas = require("../../utils/tool_schemas");
var _handler = require("./handler");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const OBSERVABILITY_RUN_LOG_RATE_ANALYSIS_TOOL_ID = exports.OBSERVABILITY_RUN_LOG_RATE_ANALYSIS_TOOL_ID = 'observability.run_log_rate_analysis';
const logRateAnalysisSchema = _zod.z.object({
  index: _zod.z.string().describe(_tool_schemas.indexDescription),
  timeFieldName: _zod.z.string().describe('Timestamp field used to build the baseline/deviation windows. Defaults to `@timestamp`.').optional(),
  baseline: _zod.z.object(_tool_schemas.timeRangeSchemaRequired).describe('Time range representing "normal" behavior that the deviation window will be compared against.'),
  deviation: _zod.z.object(_tool_schemas.timeRangeSchemaRequired).describe('Time range representing the time period with unusual behavior.'),
  searchQuery: _zod.z.record(_zod.z.any()).describe('Optional Elasticsearch query DSL filter that limits which documents are analyzed. Defaults to a match_all query.').optional()
});
function createRunLogRateAnalysisTool({
  core,
  logger
}) {
  const toolDefinition = {
    id: OBSERVABILITY_RUN_LOG_RATE_ANALYSIS_TOOL_ID,
    type: _onechatCommon.ToolType.builtin,
    description: `Analyzes which log fields or message patterns correlate with changes in log throughput (spikes or drops).

When to use:
- Log volume suddenly increased or decreased and you want to know WHY
- Identifying which services, hosts, or error types are driving a spike
- Correlating throughput changes to specific log categories or field values
- Answering "what changed?" when looking at log rate anomalies

How it works:
Compares a baseline time window to a deviation window and performs statistical correlation analysis to find fields/patterns associated with the change.

Do NOT use for:
- Understanding the sequence of events for a specific error (use get_correlated_logs)
- Getting a general overview of log types (use get_log_categories)
- Investigating individual log entries or transactions`,
    schema: logRateAnalysisSchema,
    tags: ['observability', 'logs'],
    availability: {
      cacheMode: 'space',
      handler: async ({
        request
      }) => {
        return (0, _get_agent_builder_resource_availability.getAgentBuilderResourceAvailability)({
          core,
          request,
          logger
        });
      }
    },
    handler: async (toolParams, context) => {
      const {
        index,
        timeFieldName = '@timestamp',
        baseline,
        deviation,
        searchQuery
      } = toolParams;
      try {
        const esClient = context.esClient.asCurrentUser;
        const {
          analysisType,
          items
        } = await (0, _handler.getToolHandler)({
          esClient,
          logger,
          index,
          timeFieldName,
          baseline,
          deviation,
          searchQuery
        });
        return {
          results: [{
            type: _tool_result.ToolResultType.other,
            data: {
              analysisType,
              items
            }
          }]
        };
      } catch (error) {
        logger.error(`Log rate analysis tool failed: ${error.message}`);
        return {
          results: [{
            type: _tool_result.ToolResultType.error,
            data: {
              message: error.message,
              stack: error.stack
            }
          }]
        };
      }
    }
  };
  return toolDefinition;
}