"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.OBSERVABILITY_GET_TRACE_METRICS_TOOL_ID = void 0;
exports.createGetTraceMetricsTool = createGetTraceMetricsTool;
var _zod = require("@kbn/zod");
var _onechatCommon = require("@kbn/onechat-common");
var _tool_result = require("@kbn/onechat-common/tools/tool_result");
var _tool_schemas = require("../../utils/tool_schemas");
var _get_agent_builder_resource_availability = require("../../utils/get_agent_builder_resource_availability");
var _handler = require("./handler");
var _ = require("..");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const OBSERVABILITY_GET_TRACE_METRICS_TOOL_ID = exports.OBSERVABILITY_GET_TRACE_METRICS_TOOL_ID = 'observability.get_trace_metrics';
const getTraceMetricsSchema = _zod.z.object({
  ..._tool_schemas.timeRangeSchemaRequired,
  kqlFilter: _zod.z.string().optional().describe('KQL filter to scope the data. Examples: \'service.name: "frontend"\', \'service.name: "checkout" AND transaction.name: "POST /api/cart"\', \'host.name: "web-*"\'.'),
  groupBy: _zod.z.string().optional().describe('Field to group results by. Common fields: "service.name", "transaction.name", "host.name", "container.id". Use low-cardinality fields where possible for meaningful aggregations. If not specified, results are grouped by service.name.')
});
function createGetTraceMetricsTool({
  core,
  dataRegistry,
  logger
}) {
  const toolDefinition = {
    id: OBSERVABILITY_GET_TRACE_METRICS_TOOL_ID,
    type: _onechatCommon.ToolType.builtin,
    description: `Retrieves trace metrics (throughput, failure rate, latency) for APM data with flexible filtering and grouping. 
        
Trace metrics are:
- Throughput: requests per minute
- Failure rate: percentage of failed transactions (0-1)
- Latency: average response time in milliseconds

Transaction types:
This tool includes ALL transaction types by default, unlike ${_.OBSERVABILITY_GET_SERVICES_TOOL_ID} which only shows the primary transaction type per service.
Default transaction types are: "request", "page-load", "mobile".
To filter by transaction type, use the kqlFilter parameter: kqlFilter='transaction.type: "request"'.

When to use this tool:
- After identifying an unhealthy service with the ${_.OBSERVABILITY_GET_SERVICES_TOOL_ID} tool, use this tool to drill down and find the root cause
- Analyze which specific transactions, hosts, or containers are causing performance issues
- Compare trace metrics across different dimensions (e.g., by transaction name, host, region)

When NOT to use this tool:
- For a high-level overview of all services, use the ${_.OBSERVABILITY_GET_SERVICES_TOOL_ID} tool instead
- For infrastructure metrics (CPU, memory, disk), use the ${_.OBSERVABILITY_GET_HOSTS_TOOL_ID} tool instead

Example workflow:
1. Call the ${_.OBSERVABILITY_GET_SERVICES_TOOL_ID} tool to identify a service with high latency
2. Call get_trace_metrics(kqlFilter='service.name: "frontend"', groupBy='transaction.name') to find which transaction is slow
3. Call get_trace_metrics(kqlFilter='service.name: "frontend" AND transaction.name: "POST /api/cart"', groupBy='host.name') to identify if specific hosts are affected

Returns an array of items with: group (the groupBy field value), latency (ms), throughput (rpm), failureRate (0-1).`,
    schema: getTraceMetricsSchema,
    tags: ['observability', 'services', 'trace', 'metrics'],
    availability: {
      cacheMode: 'space',
      handler: async ({
        request
      }) => {
        return (0, _get_agent_builder_resource_availability.getAgentBuilderResourceAvailability)({
          core,
          request,
          logger
        });
      }
    },
    handler: async ({
      start,
      end,
      kqlFilter,
      groupBy
    }, context) => {
      const {
        request
      } = context;
      const groupByField = groupBy || 'service.name';
      try {
        const {
          items
        } = await (0, _handler.getToolHandler)({
          request,
          dataRegistry,
          start,
          end,
          kqlFilter,
          groupBy: groupByField
        });
        return {
          results: [{
            type: _tool_result.ToolResultType.other,
            data: {
              items
            }
          }]
        };
      } catch (error) {
        logger.error(`Error getting trace metrics: ${error.message}`);
        logger.debug(error);
        return {
          results: [{
            type: _tool_result.ToolResultType.error,
            data: {
              message: `Failed to fetch trace metrics: ${error.message}`,
              stack: error.stack
            }
          }]
        };
      }
    }
  };
  return toolDefinition;
}