"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.OBSERVABILITY_GET_LOG_CHANGE_POINTS_TOOL_ID = void 0;
exports.createGetLogChangePointsTool = createGetLogChangePointsTool;
var _zod = require("@kbn/zod");
var _onechatCommon = require("@kbn/onechat-common");
var _tool_result = require("@kbn/onechat-common/tools/tool_result");
var _get_agent_builder_resource_availability = require("../../utils/get_agent_builder_resource_availability");
var _tool_schemas = require("../../utils/tool_schemas");
var _get_logs_indices = require("../../utils/get_logs_indices");
var _handler = require("./handler");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const OBSERVABILITY_GET_LOG_CHANGE_POINTS_TOOL_ID = exports.OBSERVABILITY_GET_LOG_CHANGE_POINTS_TOOL_ID = 'observability.get_log_change_points';
const getLogChangePointsSchema = _zod.z.object({
  ..._tool_schemas.timeRangeSchemaRequired,
  index: _zod.z.string().describe('The index or index pattern to find the logs').optional(),
  kqlFilter: _zod.z.string().describe('A KQL query to filter the log documents. Examples: level:error, service.name:"my-service"').optional(),
  messageField: _zod.z.string().describe('The unstructured text field to run the categorize_text aggregation on. This groups similar logs into patterns. Defaults to message').optional()
});
function createGetLogChangePointsTool({
  core,
  plugins,
  logger
}) {
  const toolDefinition = {
    id: OBSERVABILITY_GET_LOG_CHANGE_POINTS_TOOL_ID,
    type: _onechatCommon.ToolType.builtin,
    description: `Analyzes logs to detect statistically significant changes in log message patterns over time.

When to use:
- Detecting significant changes in log message categories (spike, dip, step change, trend change, distribution change, stationary/non‑stationary, indeterminable) and identifying when they occur.

How it works:
Uses "categorize_text" aggregation to group similar unstructured messages into patterns, then detects change points (spikes, dips, trend changes) within each category.`,
    schema: getLogChangePointsSchema,
    tags: ['observability', 'logs'],
    availability: {
      cacheMode: 'space',
      handler: async ({
        request
      }) => {
        return (0, _get_agent_builder_resource_availability.getAgentBuilderResourceAvailability)({
          core,
          request,
          logger
        });
      }
    },
    handler: async ({
      start,
      end,
      index,
      kqlFilter,
      messageField = 'message'
    }, {
      esClient
    }) => {
      try {
        const logIndexPatterns = await (0, _get_logs_indices.getLogsIndices)({
          core,
          logger
        });
        const topLogChangePoints = await (0, _handler.getToolHandler)({
          esClient,
          start,
          end,
          index: index || logIndexPatterns.join(','),
          kqlFilter,
          messageField
        });
        return {
          results: [{
            type: _tool_result.ToolResultType.other,
            data: {
              changePoints: topLogChangePoints
            }
          }]
        };
      } catch (error) {
        logger.error(`Error getting log change points: ${error.message}`);
        logger.debug(error);
        return {
          results: [{
            type: _tool_result.ToolResultType.error,
            data: {
              message: `Error getting log change points: ${error.message}`,
              stack: error.stack
            }
          }]
        };
      }
    }
  };
  return toolDefinition;
}