"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.OBSERVABILITY_GET_INDEX_INFO_TOOL_ID = void 0;
exports.createGetIndexInfoTool = createGetIndexInfoTool;
var _zod = require("@kbn/zod");
var _onechatCommon = require("@kbn/onechat-common");
var _tool_result = require("@kbn/onechat-common/tools/tool_result");
var _dedent = _interopRequireDefault(require("dedent"));
var _get_agent_builder_resource_availability = require("../../utils/get_agent_builder_resource_availability");
var _tool_schemas = require("../../utils/tool_schemas");
var _get_index_overview_handler = require("./get_index_overview_handler");
var _get_index_fields_handler = require("./get_index_fields_handler");
var _get_field_values_handler = require("./get_field_values_handler");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const OBSERVABILITY_GET_INDEX_INFO_TOOL_ID = exports.OBSERVABILITY_GET_INDEX_INFO_TOOL_ID = 'observability.get_index_info';
const getIndexInfoSchema = _zod.z.object({
  operation: _zod.z.enum(['get-index-patterns', 'list-fields', 'get-field-values']).describe((0, _dedent.default)(`Operation to perform:
        - "get-index-patterns": Get observability index patterns and discovered data streams 
        - "list-fields": List names of populated fields in an index (requires: index)
        - "get-field-values": Get values for specific fields (requires: index, fields)`)),
  index: _zod.z.string().optional().describe('Index pattern (e.g., "logs-*", "metrics-*"). Required for "list-fields" and "get-field-values".'),
  fields: _zod.z.array(_zod.z.string()).max(10).optional().describe('Array of field names or wildcard patterns to get values for (e.g., ["host.name"], ["attributes.app.*"]). Required for "get-field-values".'),
  ...(0, _tool_schemas.timeRangeSchemaOptional)({
    start: 'now-24h',
    end: 'now'
  }),
  kqlFilter: _zod.z.string().optional().describe('KQL filter to scope field discovery (e.g., ["service.name: checkout"]).'),
  intent: _zod.z.string().optional().describe('Investigation focus to filter relevant fields (e.g., "memory issues", "high latency").')
});
function createGetIndexInfoTool({
  core,
  plugins,
  logger
}) {
  const toolDefinition = {
    id: OBSERVABILITY_GET_INDEX_INFO_TOOL_ID,
    type: _onechatCommon.ToolType.builtin,
    description: (0, _dedent.default)(`
      Discovers observability index patterns, fields, and field values in the user's Elasticsearch cluster.

      **When to use:**
      - Before calling tools with "kqlFilter" param to discover valid fields and values 
      - To discover custom fields available beyond standard ECS or OTel fields
      - To understand which fields have data in the index
      - To understand the sample values and ranges for fields

      **When NOT to use:**
      - When you already know the field names and values you need

      **Examples:**
      - getIndexInfo(operation: "get-index-patterns" )
      - getIndexInfo(operation: "list-fields", index: "logs-*" )
      - getIndexInfo(operation: "get-field-values", index: "logs-*", fields: ["host.name"] )
    `),
    schema: getIndexInfoSchema,
    tags: ['observability', 'index', 'fields'],
    availability: {
      cacheMode: 'space',
      handler: async ({
        request
      }) => {
        return (0, _get_agent_builder_resource_availability.getAgentBuilderResourceAvailability)({
          core,
          request,
          logger
        });
      }
    },
    handler: async (params, {
      esClient,
      modelProvider
    }) => {
      try {
        let result;
        switch (params.operation) {
          case 'get-index-patterns':
            result = await (0, _get_index_overview_handler.getIndexPatternsHandler)({
              core,
              plugins,
              esClient,
              logger
            });
            break;
          case 'list-fields':
            if (!params.index) {
              return {
                results: [{
                  type: _tool_result.ToolResultType.error,
                  data: {
                    message: '"index" is required for operation "list-fields"'
                  }
                }]
              };
            }
            result = await (0, _get_index_fields_handler.listFieldsHandler)({
              esClient,
              index: params.index,
              intent: params.intent,
              start: params.start,
              end: params.end,
              kqlFilter: params.kqlFilter,
              modelProvider,
              logger
            });
            break;
          case 'get-field-values':
            if (!params.index || !params.fields) {
              return {
                results: [{
                  type: _tool_result.ToolResultType.error,
                  data: {
                    message: '"index" and "fields" are required for operation "get-field-values"'
                  }
                }]
              };
            }
            result = await (0, _get_field_values_handler.getFieldValuesHandler)({
              esClient,
              index: params.index,
              fields: params.fields,
              start: params.start,
              end: params.end,
              kqlFilter: params.kqlFilter
            });
            break;
        }
        return {
          results: [{
            type: _tool_result.ToolResultType.other,
            data: {
              ...result
            }
          }]
        };
      } catch (error) {
        logger.error(`Error getting index info: ${error.message}`);
        return {
          results: [{
            type: _tool_result.ToolResultType.error,
            data: {
              message: `Failed to get index info: ${error.message}`
            }
          }]
        };
      }
    }
  };
  return toolDefinition;
}