"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getFieldValuesHandler = getFieldValuesHandler;
var _lodash = require("lodash");
var _get_typed_search = require("../../utils/get_typed_search");
var _dsl_filters = require("../../utils/dsl_filters");
var _time = require("../../utils/time");
var _get_field_type = require("./get_field_type");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const MAX_KEYWORD_VALUES = 50;
const MAX_TEXT_SAMPLES = 5;
const MAX_CHARS_PER_SAMPLE = 500;
const KEYWORD_TYPES = ['keyword', 'constant_keyword', 'ip'];
const NUMERIC_TYPES = ['long', 'integer', 'short', 'byte', 'double', 'float', 'half_float', 'scaled_float', 'unsigned_long'];
const DATE_TYPES = ['date', 'date_nanos'];
const BOOLEAN_TYPES = ['boolean'];
const TEXT_TYPES = ['text', 'match_only_text'];
/** Gets distinct values for a single keyword field via termsEnum with optional filtering */
async function getKeywordFieldValues(esClient, index, field, queryFilter) {
  try {
    const {
      terms
    } = await esClient.asCurrentUser.termsEnum({
      index,
      field,
      size: MAX_KEYWORD_VALUES + 1,
      index_filter: queryFilter
    });
    return {
      type: 'keyword',
      field,
      values: terms.slice(0, MAX_KEYWORD_VALUES),
      hasMoreValues: terms.length > MAX_KEYWORD_VALUES
    };
  } catch (error) {
    return {
      type: 'error',
      field,
      message: error.message
    };
  }
}

/** Gets min/max for multiple numeric fields in a single request */
async function getNumericFieldValuesBatch(esClient, index, fields, queryFilter) {
  if (fields.length === 0) return {};
  const search = (0, _get_typed_search.getTypedSearch)(esClient.asCurrentUser);
  const response = await search({
    index,
    size: 0,
    track_total_hits: false,
    query: queryFilter,
    aggs: Object.fromEntries(fields.map(field => [field, {
      stats: {
        field
      }
    }]))
  });
  return (0, _lodash.keyBy)(fields.map(field => {
    var _response$aggregation;
    const stats = (_response$aggregation = response.aggregations) === null || _response$aggregation === void 0 ? void 0 : _response$aggregation[field];
    if ((stats === null || stats === void 0 ? void 0 : stats.min) != null && (stats === null || stats === void 0 ? void 0 : stats.max) != null) {
      return {
        type: 'numeric',
        field,
        min: stats.min,
        max: stats.max
      };
    }
    return {
      type: 'error',
      field,
      message: 'No numeric values found'
    };
  }), 'field');
}

/** Gets min/max for multiple date fields in a single request */
async function getDateFieldValuesBatch(esClient, index, fields, queryFilter) {
  if (fields.length === 0) return {};
  const aggs = Object.fromEntries(fields.flatMap(field => [[`${field}_min`, {
    min: {
      field
    }
  }], [`${field}_max`, {
    max: {
      field
    }
  }]]));
  const search = (0, _get_typed_search.getTypedSearch)(esClient.asCurrentUser);
  const response = await search({
    index,
    size: 0,
    track_total_hits: false,
    query: queryFilter,
    aggs
  });
  return (0, _lodash.keyBy)(fields.map(field => {
    var _response$aggregation2, _response$aggregation3;
    const minAgg = (_response$aggregation2 = response.aggregations) === null || _response$aggregation2 === void 0 ? void 0 : _response$aggregation2[`${field}_min`];
    const maxAgg = (_response$aggregation3 = response.aggregations) === null || _response$aggregation3 === void 0 ? void 0 : _response$aggregation3[`${field}_max`];
    if (minAgg !== null && minAgg !== void 0 && minAgg.value_as_string && maxAgg !== null && maxAgg !== void 0 && maxAgg.value_as_string) {
      return {
        type: 'date',
        field,
        min: minAgg.value_as_string,
        max: maxAgg.value_as_string
      };
    }
    return {
      type: 'error',
      field,
      message: 'No date values found'
    };
  }), 'field');
}

/** Gets sample values for multiple text fields in a single request */
async function getTextFieldSampleValues(esClient, index, fields, queryFilter) {
  if (fields.length === 0) return {};
  const response = await esClient.asCurrentUser.search({
    index,
    size: MAX_TEXT_SAMPLES,
    track_total_hits: false,
    query: queryFilter,
    _source: false,
    fields
  });
  return Object.fromEntries(fields.map(field => {
    const samples = response.hits.hits.flatMap(hit => {
      var _hit$fields$field, _hit$fields;
      return (_hit$fields$field = (_hit$fields = hit.fields) === null || _hit$fields === void 0 ? void 0 : _hit$fields[field]) !== null && _hit$fields$field !== void 0 ? _hit$fields$field : [];
    }).filter(v => typeof v === 'string').slice(0, MAX_TEXT_SAMPLES).map(v => v.length > MAX_CHARS_PER_SAMPLE ? v.slice(0, MAX_CHARS_PER_SAMPLE) + '…' : v);
    if (samples.length === 0) {
      return [field, {
        type: 'error',
        field,
        message: 'No text values found'
      }];
    }
    return [field, {
      type: 'text',
      field,
      samples
    }];
  }));
}

/** Converts a wildcard pattern to a regex */
function wildcardToRegex(pattern) {
  return new RegExp(`^${pattern.replace(/\./g, '\\.').replace(/\*/g, '.*')}$`);
}

/** Determines the category for a field type */
function getFieldCategory(fieldType) {
  if (KEYWORD_TYPES.includes(fieldType)) return 'keyword';
  if (NUMERIC_TYPES.includes(fieldType)) return 'numeric';
  if (DATE_TYPES.includes(fieldType)) return 'date';
  if (BOOLEAN_TYPES.includes(fieldType)) return 'boolean';
  if (TEXT_TYPES.includes(fieldType)) return 'text';
  return 'unsupported';
}
/** Resolves an input (field name or wildcard) to concrete fields or an error */
function resolveInputToConcreteFields(input, allFieldNames, fieldNameToTypeMap) {
  const isWildcard = input.includes('*');
  const matchingFields = isWildcard ? allFieldNames.filter(f => wildcardToRegex(input).test(f) && fieldNameToTypeMap[f]) : fieldNameToTypeMap[input] ? [input] : [];
  if (matchingFields.length === 0) {
    return [{
      input,
      error: isWildcard ? `No fields match pattern "${input}"` : `Field "${input}" not found`
    }];
  }
  return matchingFields.map(field => ({
    field,
    fieldType: fieldNameToTypeMap[field],
    category: getFieldCategory(fieldNameToTypeMap[field])
  }));
}

/**
 * Field value discovery - returns values/ranges for multiple fields.
 * Batches requests by field type to minimize ES calls.
 * Supports wildcard patterns in field names (e.g., "attributes.*").
 */
async function getFieldValuesHandler({
  esClient,
  index,
  fields,
  start,
  end,
  kqlFilter
}) {
  var _byCategory$keyword, _byCategory$numeric, _byCategory$date, _byCategory$boolean, _byCategory$text, _byCategory$unsupport;
  const queryFilter = {
    bool: {
      filter: [...(0, _dsl_filters.timeRangeFilter)('@timestamp', {
        start: (0, _time.parseDatemath)(start),
        end: (0, _time.parseDatemath)(end, {
          roundUp: true
        })
      }), ...(0, _dsl_filters.kqlFilter)(kqlFilter)]
    }
  };

  // fieldCaps expands wildcards and returns field types
  // Note: also returns parent object fields from passthrough types (filtered out below)
  const capsResponse = await esClient.asCurrentUser.fieldCaps({
    index,
    fields,
    ignore_unavailable: true,
    allow_no_indices: true,
    index_filter: queryFilter
  });

  // Map field names to their concrete types (undefined for object/nested/unmapped)
  const fieldNameToTypeMap = (0, _lodash.mapValues)(capsResponse.fields, _get_field_type.getFieldType);
  const allFieldNames = Object.keys(fieldNameToTypeMap);

  // Resolve all inputs to concrete fields or errors
  const concreteFields = fields.flatMap(input => resolveInputToConcreteFields(input, allFieldNames, fieldNameToTypeMap));
  const errors = concreteFields.filter(r => 'error' in r);
  const validFields = concreteFields.filter(r => 'field' in r);

  // Group valid fields by category
  const byCategory = (0, _lodash.groupBy)(validFields, r => r.category);
  const keywordFields = ((_byCategory$keyword = byCategory.keyword) !== null && _byCategory$keyword !== void 0 ? _byCategory$keyword : []).map(r => r.field);
  const numericFields = ((_byCategory$numeric = byCategory.numeric) !== null && _byCategory$numeric !== void 0 ? _byCategory$numeric : []).map(r => r.field);
  const dateFields = ((_byCategory$date = byCategory.date) !== null && _byCategory$date !== void 0 ? _byCategory$date : []).map(r => r.field);
  const booleanFields = ((_byCategory$boolean = byCategory.boolean) !== null && _byCategory$boolean !== void 0 ? _byCategory$boolean : []).map(r => r.field);
  const textFields = ((_byCategory$text = byCategory.text) !== null && _byCategory$text !== void 0 ? _byCategory$text : []).map(r => r.field);
  const unsupportedFields = (_byCategory$unsupport = byCategory.unsupported) !== null && _byCategory$unsupport !== void 0 ? _byCategory$unsupport : [];

  // Fetch values in parallel by type
  const [keywordResults, numericResults, dateResults, textResults] = await Promise.all([Promise.all(keywordFields.map(f => getKeywordFieldValues(esClient, index, f, queryFilter))), getNumericFieldValuesBatch(esClient, index, numericFields, queryFilter), getDateFieldValuesBatch(esClient, index, dateFields, queryFilter), getTextFieldSampleValues(esClient, index, textFields, queryFilter)]);
  const errorResults = Object.fromEntries(errors.map(e => [e.input, {
    type: 'error',
    field: e.input,
    message: e.error
  }]));
  const unsupportedResults = Object.fromEntries(unsupportedFields.map(r => [r.field, {
    type: 'unsupported',
    field: r.field,
    fieldType: r.fieldType
  }]));
  const booleanResults = Object.fromEntries(booleanFields.map(f => [f, {
    type: 'boolean',
    field: f
  }]));
  return {
    fields: {
      ...errorResults,
      ...unsupportedResults,
      ...booleanResults,
      ...(0, _lodash.keyBy)(keywordResults, 'field'),
      ...numericResults,
      ...dateResults,
      ...textResults
    }
  };
}