"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getDataStreamsHandler = getDataStreamsHandler;
var _lodash = require("lodash");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/** Information about an individual data stream */

/**
 * Extracts the dataset from a data stream name.
 * Data stream names follow the pattern: {type}-{dataset}-{namespace}
 * e.g., "metrics-system.memory-default" -> "system.memory"
 */
function extractDataset(name) {
  const parts = name.split('-');
  return parts.slice(1, -1).join('-');
}

/**
 * Discovers observability data streams in the cluster.
 * Returns a flat list of data streams with their datasets, sorted by name.
 *
 * Uses the configured observability index patterns (from getObservabilityDataSources)
 * to ensure consistency and support for CCS (Cross-Cluster Search) if configured.
 *
 * @example
 * // Returns:
 * [
 *   { name: "logs-apm.error-default", dataset: "apm.error" },
 *   { name: "metrics-system.cpu-default", dataset: "system.cpu" },
 *   { name: "traces-apm-default", dataset: "apm" }
 * ]
 */
async function getDataStreamsHandler({
  esClient,
  dataSources,
  logger
}) {
  try {
    // Build pattern from configured observability index patterns (supports CCS)
    const indexPatterns = [...dataSources.logIndexPatterns, ...dataSources.metricIndexPatterns, dataSources.apmIndexPatterns.transaction, dataSources.apmIndexPatterns.span].join(',');
    const response = await esClient.asCurrentUser.indices.getDataStream({
      name: indexPatterns
    });
    return (0, _lodash.sortBy)(response.data_streams.map(ds => ({
      name: ds.name,
      dataset: extractDataset(ds.name)
    })), 'name');
  } catch (error) {
    logger.error(`Error retrieving data streams: ${error}`);
    return [];
  }
}