"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.OBSERVABILITY_GET_DOWNSTREAM_DEPENDENCIES_TOOL_ID = void 0;
exports.createDownstreamDependenciesTool = createDownstreamDependenciesTool;
var _zod = require("@kbn/zod");
var _onechatCommon = require("@kbn/onechat-common");
var _tool_result = require("@kbn/onechat-common/tools/tool_result");
var _tool_schemas = require("../../utils/tool_schemas");
var _get_agent_builder_resource_availability = require("../../utils/get_agent_builder_resource_availability");
var _handler = require("./handler");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const OBSERVABILITY_GET_DOWNSTREAM_DEPENDENCIES_TOOL_ID = exports.OBSERVABILITY_GET_DOWNSTREAM_DEPENDENCIES_TOOL_ID = 'observability.get_downstream_dependencies';
const DEFAULT_TIME_RANGE = {
  start: 'now-1h',
  end: 'now'
};
const getDownstreamDependenciesToolSchema = _zod.z.object({
  ...(0, _tool_schemas.timeRangeSchemaOptional)(DEFAULT_TIME_RANGE),
  serviceName: _zod.z.string().min(1).describe('The name of the service'),
  serviceEnvironment: _zod.z.string().optional().describe('The environment that the service is running in. Leave empty to query for all environments.')
});
function createDownstreamDependenciesTool({
  core,
  dataRegistry,
  logger
}) {
  const toolDefinition = {
    id: OBSERVABILITY_GET_DOWNSTREAM_DEPENDENCIES_TOOL_ID,
    type: _onechatCommon.ToolType.builtin,
    description: `Identifies downstream dependencies (other services, databases, external APIs) for a specific service.

When to use:
- Showing the topology for a service
- Investigating if a dependency is causing issues
- Understanding the blast radius of a failing service`,
    schema: getDownstreamDependenciesToolSchema,
    tags: ['observability', 'apm', 'dependencies'],
    availability: {
      cacheMode: 'space',
      handler: async ({
        request
      }) => {
        return (0, _get_agent_builder_resource_availability.getAgentBuilderResourceAvailability)({
          core,
          request,
          logger
        });
      }
    },
    handler: async (toolParams, context) => {
      const {
        serviceName,
        serviceEnvironment,
        start,
        end
      } = toolParams;
      const {
        request
      } = context;
      try {
        var _dependencies$length;
        const {
          dependencies
        } = await (0, _handler.getToolHandler)({
          request,
          dataRegistry,
          serviceName,
          serviceEnvironment,
          start,
          end
        });
        const total = (_dependencies$length = dependencies === null || dependencies === void 0 ? void 0 : dependencies.length) !== null && _dependencies$length !== void 0 ? _dependencies$length : 0;
        return {
          results: [{
            type: _tool_result.ToolResultType.other,
            data: {
              total,
              dependencies
            }
          }]
        };
      } catch (error) {
        logger.error(`Error getting APM downstream dependencies: ${error.message}`);
        logger.debug(error);
        return {
          results: [{
            type: _tool_result.ToolResultType.error,
            data: {
              message: `Failed to fetch downstream dependencies: ${error.message}`,
              stack: error.stack
            }
          }]
        };
      }
    }
  };
  return toolDefinition;
}