"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.OBSERVABILITY_GET_ANOMALY_DETECTION_JOBS_TOOL_ID = void 0;
exports.createGetAnomalyDetectionJobsTool = createGetAnomalyDetectionJobsTool;
var _zod = require("@kbn/zod");
var _onechatCommon = require("@kbn/onechat-common");
var _tool_result = require("@kbn/onechat-common/tools/tool_result");
var _get_agent_builder_resource_availability = require("../../utils/get_agent_builder_resource_availability");
var _tool_schemas = require("../../utils/tool_schemas");
var _handler = require("./handler");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const OBSERVABILITY_GET_ANOMALY_DETECTION_JOBS_TOOL_ID = exports.OBSERVABILITY_GET_ANOMALY_DETECTION_JOBS_TOOL_ID = 'observability.get_anomaly_detection_jobs';
const DEFAULT_JOBS_LIMIT = 10;
const DEFAULT_TIME_RANGE = {
  start: 'now-24h',
  end: 'now'
};
const getAnomalyDetectionJobsSchema = _zod.z.object({
  jobIds: _zod.z.array(_zod.z.string().min(1)).min(1).max(20).describe('Optional list of ML job IDs to query. Leave empty to include all anomaly detection jobs in this space.').optional(),
  limit: _zod.z.number().int().min(1).max(25).describe(`Maximum number of jobs to return. Defaults to ${DEFAULT_JOBS_LIMIT}.`).optional(),
  ...(0, _tool_schemas.timeRangeSchemaOptional)(DEFAULT_TIME_RANGE)
});
function createGetAnomalyDetectionJobsTool({
  core,
  plugins,
  logger
}) {
  const toolDefinition = {
    id: OBSERVABILITY_GET_ANOMALY_DETECTION_JOBS_TOOL_ID,
    type: _onechatCommon.ToolType.builtin,
    description: `Retrieves Machine Learning anomaly detection jobs and their top anomaly records.

When to use:
- Investigating anomalies in logs, metrics, or traces
- Finding outliers that might indicate problems
- Answering "is anything behaving abnormally?"`,
    schema: getAnomalyDetectionJobsSchema,
    tags: ['observability', 'machine_learning', 'anomaly_detection'],
    availability: {
      cacheMode: 'space',
      handler: async ({
        request
      }) => {
        return (0, _get_agent_builder_resource_availability.getAgentBuilderResourceAvailability)({
          core,
          request,
          logger
        });
      }
    },
    handler: async (toolParams, {
      esClient,
      request
    }) => {
      const {
        jobIds,
        limit: jobsLimit = DEFAULT_JOBS_LIMIT,
        start: rangeStart,
        end: rangeEnd
      } = toolParams;
      const scopedEsClient = esClient.asCurrentUser;
      const mlClient = scopedEsClient.ml;
      try {
        const mlJobs = await (0, _handler.getToolHandler)({
          core,
          plugins,
          mlClient,
          request,
          logger,
          jobIds,
          jobsLimit,
          rangeStart,
          rangeEnd
        });
        return {
          results: [{
            type: _tool_result.ToolResultType.other,
            data: {
              jobs: mlJobs,
              total: mlJobs.length
            }
          }]
        };
      } catch (error) {
        logger.error(`Error retrieving anomaly detection jobs: ${error.message}`);
        logger.debug(error);
        return {
          results: [{
            type: _tool_result.ToolResultType.error,
            data: {
              message: `Failed to retrieve anomaly detection jobs: ${error.message}`,
              stack: error.stack
            }
          }]
        };
      }
    }
  };
  return toolDefinition;
}