"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.emsAutoSuggest = emsAutoSuggest;
exports.suggestEMSTermJoinConfig = suggestEMSTermJoinConfig;
var _util = require("../util");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function suggestEMSTermJoinConfig(sampleValuesConfig) {
  try {
    const fileLayers = await (0, _util.getEmsFileLayers)();
    return emsAutoSuggest(sampleValuesConfig, fileLayers);
  } catch (error) {
    // can not return suggestions since EMS is not available.
    return null;
  }
}
function emsAutoSuggest(sampleValuesConfig, fileLayers) {
  const matches = [];
  if (sampleValuesConfig.sampleValuesColumnName) {
    const matchesBasedOnColumnName = suggestByName(fileLayers, sampleValuesConfig.sampleValuesColumnName, sampleValuesConfig.sampleValues);
    matches.push(...matchesBasedOnColumnName);
  }
  if (sampleValuesConfig.sampleValues && sampleValuesConfig.sampleValues.length) {
    // Only looks at id-values in main manifest
    const matchesBasedOnIds = suggestByIdValues(fileLayers, sampleValuesConfig.sampleValues);
    matches.push(...matchesBasedOnIds);
  }
  const uniqMatches = matches.reduce((accum, match) => {
    const found = accum.find(m => {
      return m.config.layerId === match.layerId && m.config.field === match.layerId;
    });
    if (found) {
      found.count += 1;
    } else {
      accum.push({
        config: match,
        count: 1
      });
    }
    return accum;
  }, []);
  uniqMatches.sort((a, b) => {
    return b.count - a.count;
  });
  return uniqMatches.length ? uniqMatches[0].config : null;
}
function suggestByName(fileLayers, columnName, sampleValues) {
  const matches = [];
  fileLayers.forEach(fileLayer => {
    const emsFields = fileLayer.getFields();
    emsFields.forEach(emsField => {
      if (!emsField.alias || !emsField.alias.length) {
        return;
      }
      const emsConfig = {
        layerId: fileLayer.getId(),
        field: emsField.id,
        displayName: fileLayer.getDisplayName()
      };
      emsField.alias.forEach(alias => {
        const regex = new RegExp(alias, 'i');
        const nameMatchesAlias = !!columnName.match(regex);
        // Check if this violates any known id-values.

        let isMatch;
        if (sampleValues) {
          if (emsField.values && emsField.values.length) {
            isMatch = nameMatchesAlias && allSamplesMatch(sampleValues, emsField.values);
          } else {
            // requires validation against sample-values but EMS provides no meta to do so.
            isMatch = false;
          }
        } else {
          isMatch = nameMatchesAlias;
        }
        if (isMatch) {
          matches.push(emsConfig);
        }
      });
    });
  });
  return matches;
}
function allSamplesMatch(sampleValues, ids) {
  for (let j = 0; j < sampleValues.length; j++) {
    const sampleValue = sampleValues[j].toString();
    if (!ids.includes(sampleValue)) {
      return false;
    }
  }
  return true;
}
function suggestByIdValues(fileLayers, sampleValues) {
  const matches = [];
  fileLayers.forEach(fileLayer => {
    const emsFields = fileLayer.getFields();
    emsFields.forEach(emsField => {
      if (!emsField.values || !emsField.values.length) {
        return;
      }
      if (allSamplesMatch(sampleValues, emsField.values)) {
        matches.push({
          layerId: fileLayer.getId(),
          field: emsField.id,
          displayName: fileLayer.getDisplayName()
        });
      }
    });
  });
  return matches;
}