"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getFromPreloaded = void 0;
exports.loadInitial = loadInitial;
var _i18n = require("@kbn/i18n");
var _ = require("..");
var _lens_slice = require("../lens_slice");
var _utils = require("../../utils");
var _editor_frame = require("../../editor_frame_service/editor_frame");
var _constants = require("../../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * This function returns a Saved object from a either a by reference or by value input
 */
const getFromPreloaded = async ({
  initialInput,
  lensServices,
  history
}) => {
  const {
    notifications,
    spaces,
    attributeService
  } = lensServices;
  let doc;
  try {
    const docFromSavedObject = await (initialInput.savedObjectId ? attributeService.loadFromLibrary(initialInput.savedObjectId) : undefined);
    if (!docFromSavedObject) {
      return {
        // @TODO: it would be nice to address this type checks once for all
        doc: {
          ...initialInput.attributes,
          type: _constants.LENS_EMBEDDABLE_TYPE
        },
        sharingSavedObjectProps: {
          outcome: 'exactMatch'
        },
        managed: false
      };
    }
    const {
      sharingSavedObjectProps,
      attributes,
      managed
    } = docFromSavedObject;
    if (spaces && (sharingSavedObjectProps === null || sharingSavedObjectProps === void 0 ? void 0 : sharingSavedObjectProps.outcome) === 'aliasMatch' && history) {
      // We found this object by a legacy URL alias from its old ID; redirect the user to the page with its new ID, preserving any URL hash
      const newObjectId = sharingSavedObjectProps.aliasTargetId; // This is always defined if outcome === 'aliasMatch'
      const newPath = lensServices.http.basePath.prepend(`${(0, _constants.getEditPath)(newObjectId)}${history.location.search}`);
      await spaces.ui.redirectLegacyUrl({
        path: newPath,
        aliasPurpose: sharingSavedObjectProps.aliasPurpose,
        objectNoun: _i18n.i18n.translate('xpack.lens.legacyUrlConflict.objectNoun', {
          defaultMessage: 'Lens visualization'
        })
      });
    }
    doc = {
      ...initialInput,
      ...attributes,
      type: _constants.LENS_EMBEDDABLE_TYPE
    };
    return {
      doc,
      sharingSavedObjectProps: {
        aliasTargetId: sharingSavedObjectProps === null || sharingSavedObjectProps === void 0 ? void 0 : sharingSavedObjectProps.aliasTargetId,
        outcome: sharingSavedObjectProps === null || sharingSavedObjectProps === void 0 ? void 0 : sharingSavedObjectProps.outcome
      },
      managed: Boolean(managed)
    };
  } catch (e) {
    notifications.toasts.addDanger(_i18n.i18n.translate('xpack.lens.app.docLoadingError', {
      defaultMessage: 'Error loading saved document'
    }));
  }
};
exports.getFromPreloaded = getFromPreloaded;
/**
 * Initialize project routing in CPS Manager
 * Determines whether to preserve or reset the project routing based on the context
 */
function initProjectRoutingState({
  cps,
  lens,
  initialInput,
  isLinkedToOriginatingApp = false
}) {
  if (!(cps !== null && cps !== void 0 && cps.cpsManager)) {
    return;
  }
  const cpsManager = cps.cpsManager;

  // Check if dataSources have been initialized already to detect save/save-as scenarios
  const hasActiveLensSession = !Object.values(lens.datasourceStates).every(ds => ds.state === null);

  // Reset when: NOT linked to originating app, NOT active session, NOT embeddable
  const shouldReset = !isLinkedToOriginatingApp && !hasActiveLensSession && !initialInput;
  if (shouldReset) {
    const defaultRouting = cpsManager.getDefaultProjectRouting();
    cpsManager.setProjectRouting(defaultRouting);
    return defaultRouting;
  } else {
    return cpsManager.getProjectRouting();
  }
}
async function loadFromLocatorState(store, initialState, loaderSharedArgs, {
  notifications,
  data
}, emptyState, autoApplyDisabled, projectRouting) {
  var _lens$persistedDoc;
  const {
    lens
  } = store.getState();
  const locatorReferences = 'references' in initialState ? initialState.references : undefined;
  const {
    datasourceStates,
    visualizationState,
    indexPatterns,
    indexPatternRefs,
    annotationGroups
  } = await (0, _editor_frame.initializeSources)({
    visualizationState: emptyState.visualization,
    datasourceStates: emptyState.datasourceStates,
    adHocDataViews: ((_lens$persistedDoc = lens.persistedDoc) === null || _lens$persistedDoc === void 0 ? void 0 : _lens$persistedDoc.state.adHocDataViews) || initialState.dataViewSpecs,
    references: locatorReferences,
    ...loaderSharedArgs
  }, {
    isFullEditor: true
  });
  const currentSessionId = (initialState === null || initialState === void 0 ? void 0 : initialState.searchSessionId) || data.search.session.getSessionId();
  store.dispatch((0, _.initExisting)({
    isSaveable: true,
    filters: initialState.filters || data.query.filterManager.getFilters(),
    query: initialState.query || emptyState.query,
    searchSessionId: currentSessionId,
    activeDatasourceId: emptyState.activeDatasourceId,
    visualization: {
      activeId: emptyState.visualization.activeId,
      state: visualizationState,
      selectedLayerId: emptyState.visualization.selectedLayerId
    },
    dataViews: (0, _utils.getInitialDataViewsObject)(indexPatterns, indexPatternRefs),
    datasourceStates: Object.entries(datasourceStates).reduce((state, [datasourceId, datasourceState]) => ({
      ...state,
      [datasourceId]: {
        ...datasourceState,
        isLoading: false
      }
    }), {}),
    isLoading: false,
    annotationGroups,
    projectRouting
  }));
  if (autoApplyDisabled) {
    store.dispatch((0, _lens_slice.disableAutoApply)());
  }
}
async function loadFromEmptyState(store, emptyState, loaderSharedArgs, {
  data
}, activeDatasourceId, autoApplyDisabled, projectRouting) {
  var _lens$persistedDoc2;
  const {
    lens
  } = store.getState();
  const {
    datasourceStates,
    indexPatterns,
    indexPatternRefs
  } = await (0, _editor_frame.initializeSources)({
    visualizationState: lens.visualization,
    datasourceStates: lens.datasourceStates,
    adHocDataViews: (_lens$persistedDoc2 = lens.persistedDoc) === null || _lens$persistedDoc2 === void 0 ? void 0 : _lens$persistedDoc2.state.adHocDataViews,
    ...loaderSharedArgs
  }, {
    isFullEditor: true
  });
  store.dispatch((0, _.initEmpty)({
    newState: {
      ...emptyState,
      dataViews: (0, _utils.getInitialDataViewsObject)(indexPatterns, indexPatternRefs),
      searchSessionId: data.search.session.getSessionId() || data.search.session.start(),
      ...(activeDatasourceId && {
        activeDatasourceId
      }),
      datasourceStates: Object.entries(datasourceStates).reduce((state, [datasourceId, datasourceState]) => ({
        ...state,
        [datasourceId]: {
          ...datasourceState,
          isLoading: false
        }
      }), {}),
      isLoading: false,
      projectRouting
    },
    initialContext: loaderSharedArgs.initialContext
  }));
  if (autoApplyDisabled) {
    store.dispatch((0, _lens_slice.disableAutoApply)());
  }
}
async function loadFromSavedObject(store, savedObjectId, persisted, loaderSharedArgs, {
  data,
  chrome
}, autoApplyDisabled, inlineEditing, projectRouting) {
  const {
    doc,
    sharingSavedObjectProps,
    managed
  } = persisted;
  if (savedObjectId) {
    chrome.recentlyAccessed.add((0, _constants.getFullPath)(savedObjectId), doc.title, savedObjectId);
  }
  const docDatasourceStates = Object.entries(doc.state.datasourceStates).reduce((stateMap, [datasourceId, datasourceState]) => ({
    ...stateMap,
    [datasourceId]: {
      isLoading: true,
      state: datasourceState
    }
  }), {});

  // when the embeddable is initialized from the dashboard we don't want to inject the filters
  // as this will replace the parent application filters (such as a dashboard)
  if (!inlineEditing) {
    const filters = data.query.filterManager.inject(doc.state.filters, doc.references);
    // Don't overwrite any pinned filters
    data.query.filterManager.setAppFilters(filters);
  }
  const docVisualizationState = {
    activeId: doc.visualizationType,
    state: doc.state.visualization,
    selectedLayerId: null
  };
  const {
    datasourceStates,
    visualizationState,
    indexPatterns,
    indexPatternRefs,
    annotationGroups
  } = await (0, _editor_frame.initializeSources)({
    visualizationState: docVisualizationState,
    datasourceStates: docDatasourceStates,
    references: [...doc.references, ...(doc.state.internalReferences || [])],
    adHocDataViews: doc.state.adHocDataViews,
    ...loaderSharedArgs
  }, {
    isFullEditor: true
  });
  const currentSessionId = data.search.session.getSessionId();
  store.dispatch((0, _.initExisting)({
    isSaveable: true,
    sharingSavedObjectProps,
    filters: data.query.filterManager.getFilters(),
    query: doc.state.query,
    searchSessionId: !savedObjectId && currentSessionId ? currentSessionId : !inlineEditing ? data.search.session.start() : undefined,
    persistedDoc: {
      ...doc,
      state: {
        ...doc.state,
        visualization: visualizationState
      }
    },
    activeDatasourceId: (0, _utils.getInitialDatasourceId)(loaderSharedArgs.datasourceMap, doc),
    visualization: {
      activeId: doc.visualizationType,
      state: visualizationState,
      selectedLayerId: null
    },
    dataViews: (0, _utils.getInitialDataViewsObject)(indexPatterns, indexPatternRefs),
    datasourceStates: Object.entries(datasourceStates).reduce((state, [datasourceId, datasourceState]) => ({
      ...state,
      [datasourceId]: {
        ...datasourceState,
        isLoading: false
      }
    }), {}),
    isLoading: false,
    annotationGroups,
    managed,
    projectRouting
  }));
  if (autoApplyDisabled) {
    store.dispatch((0, _lens_slice.disableAutoApply)());
  }
}
async function loadInitial(store, storeDeps, {
  redirectCallback,
  initialInput,
  history,
  inlineEditing
}, autoApplyDisabled) {
  var _lens$persistedDoc3;
  const {
    lensServices,
    datasourceMap,
    initialContext,
    initialStateFromLocator,
    visualizationMap
  } = storeDeps;
  const {
    resolvedDateRange,
    searchSessionId,
    isLinkedToOriginatingApp,
    ...emptyState
  } = (0, _lens_slice.getPreloadedState)(storeDeps);
  const {
    notifications,
    data,
    cps
  } = lensServices;
  const {
    lens
  } = store.getState();
  const projectRouting = initProjectRoutingState({
    cps,
    lens,
    initialInput,
    isLinkedToOriginatingApp
  });
  const loaderSharedArgs = {
    visualizationMap,
    initialContext,
    datasourceMap,
    dataViews: lensServices.dataViews,
    storage: lensServices.storage,
    eventAnnotationService: lensServices.eventAnnotationService,
    defaultIndexPatternId: lensServices.uiSettings.get('defaultIndex')
  };
  let activeDatasourceId;
  if (initialContext && 'query' in initialContext) {
    activeDatasourceId = 'textBased';
  }
  if (initialStateFromLocator) {
    const newFilters = initialStateFromLocator.filters ? structuredClone(initialStateFromLocator.filters) : undefined;
    if (newFilters) {
      data.query.filterManager.setAppFilters(newFilters);
    }
    if (initialStateFromLocator.resolvedDateRange) {
      const newTimeRange = {
        from: initialStateFromLocator.resolvedDateRange.fromDate,
        to: initialStateFromLocator.resolvedDateRange.toDate
      };
      data.query.timefilter.timefilter.setTime(newTimeRange);
    }

    // URL Reporting is using the locator params but also passing the savedObjectId
    // so be sure to not go here as there's no full snapshot URL
    if (!initialInput) {
      try {
        return loadFromLocatorState(store, initialStateFromLocator, loaderSharedArgs, lensServices, emptyState, autoApplyDisabled, projectRouting);
      } catch ({
        message
      }) {
        notifications.toasts.addDanger({
          title: message
        });
        return;
      }
    }
  }
  if (!initialInput || initialInput.savedObjectId && initialInput.savedObjectId === ((_lens$persistedDoc3 = lens.persistedDoc) === null || _lens$persistedDoc3 === void 0 ? void 0 : _lens$persistedDoc3.savedObjectId)) {
    const newFilters = initialContext && 'searchFilters' in initialContext && initialContext.searchFilters ? structuredClone(initialContext.searchFilters) : undefined;
    if (newFilters) {
      data.query.filterManager.setAppFilters(newFilters);
    }
    try {
      return loadFromEmptyState(store, emptyState, loaderSharedArgs, lensServices, activeDatasourceId, autoApplyDisabled, projectRouting);
    } catch ({
      message
    }) {
      notifications.toasts.addDanger({
        title: message
      });
      return redirectCallback === null || redirectCallback === void 0 ? void 0 : redirectCallback();
    }
  }
  try {
    const persisted = await getFromPreloaded({
      initialInput,
      lensServices,
      history
    });
    if (persisted) {
      try {
        return loadFromSavedObject(store, initialInput.savedObjectId, persisted, loaderSharedArgs, lensServices, autoApplyDisabled, inlineEditing, projectRouting);
      } catch ({
        message
      }) {
        notifications.toasts.addDanger({
          title: message
        });
      }
    } else {
      return redirectCallback === null || redirectCallback === void 0 ? void 0 : redirectCallback();
    }
  } catch (e) {
    try {
      store.dispatch((0, _.setState)({
        isLoading: false
      }));
      redirectCallback === null || redirectCallback === void 0 ? void 0 : redirectCallback();
    } catch ({
      message
    }) {
      notifications.toasts.addDanger({
        title: message
      });
      redirectCallback === null || redirectCallback === void 0 ? void 0 : redirectCallback();
    }
  }
}