"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.addLayerColumn = void 0;
exports.buildDatasetState = buildDatasetState;
exports.buildDatasetStateESQL = buildDatasetStateESQL;
exports.buildDatasetStateNoESQL = buildDatasetStateNoESQL;
exports.buildDatasourceStates = void 0;
exports.buildReferences = buildReferences;
exports.createDataViewReference = createDataViewReference;
exports.getAdhocDataviews = exports.generateLayer = exports.generateApiLayer = exports.filtersToLensState = exports.filtersToApiFormat = exports.filtersAndQueryToLensState = exports.filtersAndQueryToApiFormat = void 0;
exports.getDatasetIndex = getDatasetIndex;
exports.getDefaultReferences = void 0;
exports.isDataViewSpec = isDataViewSpec;
exports.isFormBasedLayer = isFormBasedLayer;
exports.isSingleLayer = isSingleLayer;
exports.isTextBasedLayer = isTextBasedLayer;
exports.nonNullable = nonNullable;
exports.queryToLensState = exports.queryToApiFormat = exports.operationFromColumn = void 0;
var _esqlUtils = require("@kbn/esql-utils");
var _esQuery = require("@kbn/es-query");
var _buckets = require("./columns/buckets");
var _metric = require("./columns/metric");
var _utils = require("./columns/utils");
var _constants = require("./constants");
var _constants2 = require("../schema/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

function createDataViewReference(index, layerId) {
  return {
    type: _constants.INDEX_PATTERN_ID,
    id: index,
    name: `${_constants.LENS_LAYER_SUFFIX}${layerId}`
  };
}
const getDefaultReferences = (index, dataLayerId) => {
  return [createDataViewReference(index, dataLayerId)];
};

// Need to dance a bit to satisfy TypeScript
exports.getDefaultReferences = getDefaultReferences;
function convertToTypedLayerColumns(layer) {
  // @TODO move it to satisfies
  return layer;
}

/**
 * given Lens State layer and column id, returns the corresponding Lens API operation
 * @param columnId
 * @param layer
 * @returns
 */
const operationFromColumn = (columnId, layer) => {
  const typedLayer = convertToTypedLayerColumns(layer);
  const column = typedLayer.columns[columnId];
  if (!column) return;

  // map columns to array of { column, id }
  const columnMap = Object.entries(layer.columns).map(([id, c]) => ({
    // need to cast here as the GenericIndexPatternColumn type is not compatible with Reference based column types
    column: c,
    id
  }));
  if ((0, _utils.isLensStateBucketColumnType)(column)) {
    return (0, _buckets.fromBucketLensStateToAPI)(column, columnMap);
  }
  return (0, _metric.getMetricApiColumnFromLensState)(column, typedLayer.columns);
};
exports.operationFromColumn = operationFromColumn;
function isFormBasedLayer(layer) {
  return 'columnOrder' in layer;
}
function isTextBasedLayer(layer) {
  return 'index' in layer && 'query' in layer;
}
function generateAdHocDataViewId(dataView) {
  var _dataView$timeFieldNa;
  return `${dataView.index}-${(_dataView$timeFieldNa = dataView.timeFieldName) !== null && _dataView$timeFieldNa !== void 0 ? _dataView$timeFieldNa : 'no_time_field'}`;
}
function getAdHocDataViewSpec(dataView) {
  return {
    // Improve id genertation to be more predictable and hit cache more often
    id: generateAdHocDataViewId(dataView),
    title: dataView.index,
    name: dataView.index,
    timeFieldName: dataView.timeFieldName,
    sourceFilters: [],
    fieldFormats: {},
    runtimeFieldMap: {},
    fieldAttrs: {},
    allowNoIndex: false,
    allowHidden: false
  };
}
const getAdhocDataviews = dataviews => {
  // filter out ad hoc dataViews only
  const adHocDataViewsFiltered = Object.entries(dataviews).filter(([_layerId, dataViewEntry]) => dataViewEntry.type === 'adHocDataView');
  const internalReferencesMap = new Map();

  // dedupe and map multiple layer references to the same ad hoc dataview
  for (const [layerId, dataViewEntry] of adHocDataViewsFiltered) {
    if (!internalReferencesMap.has(dataViewEntry)) {
      internalReferencesMap.set(dataViewEntry, {
        layerIds: [],
        id: generateAdHocDataViewId(dataViewEntry)
      });
    }
    const internalRef = internalReferencesMap.get(dataViewEntry);
    internalRef.layerIds.push(layerId);
  }
  const adHocDataViews = {};
  const internalReferences = [];
  for (const [baseSpec, {
    layerIds,
    id
  }] of Array.from(internalReferencesMap.entries())) {
    adHocDataViews[id] = getAdHocDataViewSpec(baseSpec);
    for (const layerId of layerIds) {
      internalReferences.push(createDataViewReference(id, layerId));
    }
  }
  return {
    adHocDataViews,
    internalReferences
  };
};
exports.getAdhocDataviews = getAdhocDataviews;
function buildDatasetStateESQL(layer) {
  var _layer$query$esql, _layer$query;
  return {
    type: 'esql',
    query: (_layer$query$esql = (_layer$query = layer.query) === null || _layer$query === void 0 ? void 0 : _layer$query.esql) !== null && _layer$query$esql !== void 0 ? _layer$query$esql : ''
  };
}
function isDataViewSpec(spec) {
  return spec != null && typeof spec === 'object' && 'title' in spec;
}
function getReferenceCriteria(layerId) {
  return ref => ref.name === `${_constants.LENS_LAYER_SUFFIX}${layerId}`;
}
function buildDatasetStateNoESQL(layer, layerId, adHocDataViews, references, adhocReferences = []) {
  var _layer$indexPatternId;
  const referenceCriteria = getReferenceCriteria(layerId);
  const adhocReference = adhocReferences === null || adhocReferences === void 0 ? void 0 : adhocReferences.find(referenceCriteria);
  if (adhocReference && adHocDataViews !== null && adHocDataViews !== void 0 && adHocDataViews[adhocReference.id]) {
    const dataViewSpec = adHocDataViews[adhocReference.id];
    if (isDataViewSpec(dataViewSpec)) {
      var _dataViewSpec$timeFie;
      return {
        type: 'index',
        index: dataViewSpec.title,
        time_field: (_dataViewSpec$timeFie = dataViewSpec.timeFieldName) !== null && _dataViewSpec$timeFie !== void 0 ? _dataViewSpec$timeFie : _constants.LENS_DEFAULT_TIME_FIELD
      };
    }
  }
  const reference = references === null || references === void 0 ? void 0 : references.find(referenceCriteria);
  if (reference) {
    return {
      type: 'dataView',
      id: reference.id
    };
  }
  return {
    type: 'dataView',
    id: 'indexPatternId' in layer ? (_layer$indexPatternId = layer.indexPatternId) !== null && _layer$indexPatternId !== void 0 ? _layer$indexPatternId : '' : ''
  };
}

/**
 * Builds dataset state from the layer configuration
 *
 * @deprecated use `buildDatasetStateESQL` or `buildDatasetStateNoESQL` instead
 */
function buildDatasetState(layer, layerId, adHocDataViews, references, adhocReferences = []) {
  if (isTextBasedLayer(layer)) {
    return buildDatasetStateESQL(layer);
  }
  return buildDatasetStateNoESQL(layer, layerId, adHocDataViews, references, adhocReferences);
}

// builds Lens State references from list of dataviews
function buildReferences(dataviews) {
  const references = [];
  for (const layerid in dataviews) {
    if (dataviews[layerid]) {
      references.push(getDefaultReferences(dataviews[layerid], layerid));
    }
  }
  return references.flat(2);
}
function isSingleLayer(layer) {
  return layer && typeof layer === 'object' && ('columnOrder' in layer || 'columns' in layer);
}

/**
 * Gets DataView from the dataset configuration
 *
 * @param dataset
 * @param dataViewsAPI
 * @returns
 */
function getDatasetIndex(dataset) {
  const timeFieldName = _constants.LENS_DEFAULT_TIME_FIELD;
  switch (dataset.type) {
    case 'index':
      return {
        index: dataset.index,
        timeFieldName
      };
    case 'esql':
      return {
        index: (0, _esqlUtils.getIndexPatternFromESQLQuery)(dataset.query),
        timeFieldName
      };
    case 'dataView':
      return {
        index: dataset.id,
        timeFieldName
      };
    case 'table':
      return;
    default:
      throw Error('dataset type not supported');
  }
}

// internal function used to build datasource states layer
function buildDatasourceStatesLayer(layer, i, dataset, datasetIndex, buildDataLayer, getValueColumns) {
  function buildValueLayer(config, ds) {
    const table = ds.table;
    const newLayer = {
      table,
      columns: getValueColumns(config, i),
      allColumns: table.columns.map(column => ({
        fieldName: column.name,
        columnId: column.id,
        meta: column.meta
      })),
      index: '',
      query: undefined
    };
    return newLayer;
  }
  function buildESQLLayer(config, ds) {
    const columns = getValueColumns(config, i);
    return {
      index: datasetIndex.index,
      query: {
        esql: ds.query
      },
      timeField: _constants.LENS_DEFAULT_TIME_FIELD,
      columns
    };
  }
  if (dataset.type === 'esql') {
    return ['textBased', buildESQLLayer(layer, dataset)];
  }
  if (dataset.type === 'table') {
    return ['textBased', buildValueLayer(layer, dataset)];
  }
  return ['formBased', buildDataLayer(layer, i, datasetIndex)];
}

/**
 * Builds lens config datasource states from LensApiState
 *
 * @param config lens api state
 * @param dataviews list to which dataviews are added
 * @param buildFormulaLayers function used when dataset type is index or dataView
 * @param getValueColumns function used when dataset type is table or esql
 * @param dataViewsAPI dataViews service
 * @returns lens datasource states
 *
 */
const buildDatasourceStates = (config, buildDataLayers, getValueColumns) => {
  let layers = {};

  // XY charts have dataset encoded per layer not at the root level
  const mainDataset = 'dataset' in config && config.dataset;
  const usedDataviews = {};
  // a few charts types support multiple layers
  const hasMultipleLayers = 'layers' in config;
  const configLayers = hasMultipleLayers ? config.layers : [config];
  for (let i = 0; i < configLayers.length; i++) {
    const layer = configLayers[i];
    const layerId = hasMultipleLayers && 'type' in layer ? `${layer.type}_${i}` : `layer_${i}`;
    const dataset = 'dataset' in layer ? layer.dataset : mainDataset;
    if (!dataset) {
      throw Error('dataset must be defined');
    }
    const index = getDatasetIndex(dataset);
    const [type, layerConfig] = buildDatasourceStatesLayer(layer, i, dataset, index, buildDataLayers, getValueColumns);
    if (layerConfig) {
      var _layers$type, _layers$type2;
      layers = {
        ...layers,
        [type]: {
          layers: isSingleLayer(layerConfig) ? {
            ...((_layers$type = layers[type]) === null || _layers$type === void 0 ? void 0 : _layers$type.layers),
            [layerId]: layerConfig
          } :
          // metric chart can return 2 layers (one for the metric and one for the trendline)
          {
            ...((_layers$type2 = layers[type]) === null || _layers$type2 === void 0 ? void 0 : _layers$type2.layers),
            ...layerConfig
          }
        }
      };

      // keep record of all dataviews used by layers
      if (index) {
        const newLayerIds = isSingleLayer(layerConfig) || Object.keys(layerConfig).length === 0 ? [layerId] : Object.keys(layerConfig);
        for (const id of newLayerIds) {
          usedDataviews[id] = dataset.type === 'dataView' ? {
            type: 'dataView',
            id: dataset.id
          } : {
            type: 'adHocDataView',
            ...index
          };
        }
      }
    }
  }
  return {
    layers,
    usedDataviews
  };
};

// adds new column to existing layer
exports.buildDatasourceStates = buildDatasourceStates;
const addLayerColumn = (layer, columnName, config, first = false, postfix = '') => {
  const [column, referenceColumn] = Array.isArray(config) ? config : [config];
  const name = columnName + postfix;
  const referenceColumnId = `${name}_reference`;
  layer.columns = {
    ...layer.columns,
    [name]: column,
    ...(referenceColumn ? {
      [referenceColumnId]: referenceColumn
    } : {})
  };
  if (first) {
    layer.columnOrder.unshift(name);
    if (referenceColumn) {
      layer.columnOrder.unshift(referenceColumnId);
    }
  } else {
    layer.columnOrder.push(name);
    if (referenceColumn) {
      layer.columnOrder.push(referenceColumnId);
    }
  }
};

/**
 * Generates the base layer
 *
 * @param id
 * @param options
 * @returns
 */
exports.addLayerColumn = addLayerColumn;
const generateLayer = (id, options) => {
  return {
    [id]: {
      sampling: options.sampling,
      ignoreGlobalFilters: options.ignore_global_filters,
      columns: {},
      columnOrder: []
    }
  };
};
exports.generateLayer = generateLayer;
const generateApiLayer = options => {
  var _options$sampling, _options$ignoreGlobal;
  if (!('columnOrder' in options)) {
    return {
      sampling: _constants2.LENS_SAMPLING_DEFAULT_VALUE,
      ignore_global_filters: _constants2.LENS_IGNORE_GLOBAL_FILTERS_DEFAULT_VALUE
    };
  }
  // mind this is already filled by schema validate
  return {
    sampling: (_options$sampling = options.sampling) !== null && _options$sampling !== void 0 ? _options$sampling : _constants2.LENS_SAMPLING_DEFAULT_VALUE,
    ignore_global_filters: (_options$ignoreGlobal = options.ignoreGlobalFilters) !== null && _options$ignoreGlobal !== void 0 ? _options$ignoreGlobal : _constants2.LENS_IGNORE_GLOBAL_FILTERS_DEFAULT_VALUE
  };
};
exports.generateApiLayer = generateApiLayer;
const filtersToApiFormat = filters => {
  return filters.map(filter => {
    var _finalFilter$query, _finalFilter$query2, _finalFilter$query$qu, _finalFilter$query3;
    const {
      $state,
      meta,
      ...finalFilter
    } = filter;
    return {
      ...((_finalFilter$query = finalFilter.query) !== null && _finalFilter$query !== void 0 && _finalFilter$query.language ? {
        language: (_finalFilter$query2 = finalFilter.query) === null || _finalFilter$query2 === void 0 ? void 0 : _finalFilter$query2.language
      } : {}),
      ...('query' in finalFilter ? {
        query: (_finalFilter$query$qu = (_finalFilter$query3 = finalFilter.query) === null || _finalFilter$query3 === void 0 ? void 0 : _finalFilter$query3.query) !== null && _finalFilter$query$qu !== void 0 ? _finalFilter$query$qu : finalFilter.query
      } : finalFilter)
    };
  });
};
exports.filtersToApiFormat = filtersToApiFormat;
const queryToApiFormat = query => {
  if (typeof query.query !== 'string') {
    return;
  }
  return {
    query: query.query,
    language: query.language
  };
};
exports.queryToApiFormat = queryToApiFormat;
const filtersToLensState = filters => {
  return filters.map(filter => {
    return {
      ...('query' in filter ? {
        query: filter.query,
        language: filter === null || filter === void 0 ? void 0 : filter.language
      } : filter),
      meta: {}
    };
  });
};
exports.filtersToLensState = filtersToLensState;
const queryToLensState = query => {
  return {
    query: query.query,
    language: query.language
  };
};
exports.queryToLensState = queryToLensState;
const filtersAndQueryToApiFormat = state => {
  var _state$state$filters;
  return {
    ...((_state$state$filters = state.state.filters) !== null && _state$state$filters !== void 0 && _state$state$filters.length ? {
      filters: filtersToApiFormat(state.state.filters)
    } : {}),
    ...(state.state.query && !(0, _esQuery.isOfAggregateQueryType)(state.state.query) ? {
      query: queryToApiFormat(state.state.query)
    } : {})
  };
};
exports.filtersAndQueryToApiFormat = filtersAndQueryToApiFormat;
function extraQueryFromAPIState(state) {
  if ('dataset' in state && state.dataset.type === 'esql') {
    return {
      esql: state.dataset.query
    };
  }
  if ('layers' in state && Array.isArray(state.layers)) {
    // pick only the first one for now
    const esqlLayer = state.layers.find(layer => {
      var _layer$dataset;
      return ((_layer$dataset = layer.dataset) === null || _layer$dataset === void 0 ? void 0 : _layer$dataset.type) === 'esql';
    });
    if (esqlLayer && 'query' in esqlLayer.dataset) {
      return {
        esql: esqlLayer.dataset.query
      };
    }
  }
  if ('query' in state && state.query) {
    return queryToLensState(state.query);
  }
  return undefined;
}
const filtersAndQueryToLensState = state => {
  const query = extraQueryFromAPIState(state);
  return {
    filters: [],
    // @TODO: rework on these with the shared Filter definition by Presentation team
    ...(state.filters ? {
      filters: filtersToLensState(state.filters)
    } : {}),
    ...(query ? {
      query
    } : {})
  };
};
exports.filtersAndQueryToLensState = filtersAndQueryToLensState;
function nonNullable(v) {
  return v != null;
}