"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.fromMetricAPItoLensState = fromMetricAPItoLensState;
exports.getMetricApiColumnFromLensState = getMetricApiColumnFromLensState;
exports.getMetricReferableApiColumnFromLensState = getMetricReferableApiColumnFromLensState;
var _uuid = require("uuid");
var _count = require("./count");
var _unique_count = require("./unique_count");
var _max_min_avg_sum = require("./max_min_avg_sum");
var _static_value = require("./static_value");
var _last_value = require("./last_value");
var _formula = require("./formula");
var _percentile = require("./percentile");
var _percentile_ranks = require("./percentile_ranks");
var _differences = require("./differences");
var _cumulative_sum = require("./cumulative_sum");
var _counter_rate = require("./counter_rate");
var _moving_average = require("./moving_average");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Specialized function signatures for transforming metric API operations to Lens state columns
 */
function fromMetricAPItoLensState(options) {
  if ((0, _utils.isAPIColumnOfType)('count', options)) {
    return [(0, _count.fromCountAPItoLensState)(options)];
  }
  if ((0, _utils.isAPIColumnOfType)('unique_count', options)) {
    return [(0, _unique_count.fromUniqueCountAPItoLensState)(options)];
  }
  if ((0, _utils.isAPIColumnOfType)('average', options) || (0, _utils.isAPIColumnOfType)('median', options) || (0, _utils.isAPIColumnOfType)('min', options) || (0, _utils.isAPIColumnOfType)('max', options) || (0, _utils.isAPIColumnOfType)('standard_deviation', options)) {
    return [(0, _max_min_avg_sum.fromBasicMetricAPItoLensState)(options)];
  }
  if ((0, _utils.isAPIColumnOfType)('sum', options)) {
    return [(0, _max_min_avg_sum.fromSumMetricAPIToLensState)(options)];
  }
  if ((0, _utils.isAPIColumnOfType)('static_value', options)) {
    return [(0, _static_value.fromStaticValueAPItoLensState)(options)];
  }
  if ((0, _utils.isAPIColumnOfType)('formula', options)) {
    return [(0, _formula.fromFormulaAPItoLensState)(options)];
  }
  if ((0, _utils.isAPIColumnOfType)('last_value', options)) {
    return [(0, _last_value.fromLastValueAPItoLensState)(options)];
  }
  if ((0, _utils.isAPIColumnOfType)('percentile', options)) {
    return [(0, _percentile.fromPercentileAPItoLensState)(options)];
  }
  if ((0, _utils.isAPIColumnOfType)('percentile_rank', options)) {
    return [(0, _percentile_ranks.fromPercentileRanksAPItoLensState)(options)];
  }
  if ((0, _utils.isAPIColumnOfType)('moving_average', options)) {
    if ((0, _utils.isApiColumnOfReferableType)(options.of)) {
      const [refColumn] = fromMetricAPItoLensState(options.of);
      return [(0, _moving_average.fromMovingAverageAPItoLensState)(options, {
        id: (0, _uuid.v4)(),
        field: options.of.field,
        label: options.of.label
      }), refColumn];
    }
  }
  if ((0, _utils.isAPIColumnOfType)('counter_rate', options)) {
    const [refColumn] = fromMetricAPItoLensState({
      operation: 'max',
      field: options.field
    });
    if (!refColumn || !('sourceField' in refColumn)) {
      return [];
    }
    return [(0, _counter_rate.fromCounterRateAPItoLensState)(options, {
      id: (0, _uuid.v4)(),
      field: refColumn.sourceField
    }), refColumn];
  }
  if ((0, _utils.isAPIColumnOfType)('cumulative_sum', options)) {
    const [refColumn] = fromMetricAPItoLensState({
      operation: 'sum',
      field: options.field,
      empty_as_null: _utils.LENS_EMPTY_AS_NULL_DEFAULT_VALUE
    });
    if (!refColumn || !('sourceField' in refColumn)) {
      return [];
    }
    return [(0, _cumulative_sum.fromCumulativeSumAPItoLensState)(options, {
      id: (0, _uuid.v4)(),
      field: refColumn.sourceField
    }), refColumn];
  }
  if ((0, _utils.isAPIColumnOfType)('differences', options)) {
    if ((0, _utils.isApiColumnOfReferableType)(options.of)) {
      const [refColumn] = fromMetricAPItoLensState(options.of);
      return [(0, _differences.fromDifferencesAPItoLensState)(options, {
        id: (0, _uuid.v4)(),
        field: refColumn.sourceField,
        label: refColumn.label
      }), refColumn];
    }
  }
  throw new Error(`Unsupported metric operation: ${options.operation}`);
}
function getMetricReferableApiColumnFromLensState(parentColumn, columns) {
  const refId = parentColumn.references[0];
  if (refId == null || columns[refId] == null) {
    throw new Error(`Missing referenced metric operation for ${parentColumn.operationType}`);
  }
  const refColumn = columns[refId];
  if ((0, _utils.isLensStateBucketColumnType)(refColumn)) {
    throw new Error(`Unsupported referenced metric operation: ${refColumn.operationType}`);
  }
  if (!(0, _utils.isColumnOfReferableType)(refColumn)) {
    throw Error(`Unsupported referable metric operation: ${refColumn.operationType}`);
  }
  const retColumn = getMetricApiColumnFromLensState(refColumn, columns);
  if (!retColumn) {
    throw new Error(`Unsupported referenced metric operation: ${refColumn.operationType}`);
  }
  return retColumn;
}
function getMetricApiColumnFromLensState(options, columns) {
  if ((0, _utils.isLensStateColumnOfType)('count', options)) {
    return (0, _count.fromCountLensStateToAPI)(options);
  }
  if ((0, _utils.isLensStateColumnOfType)('unique_count', options)) {
    return (0, _unique_count.fromUniqueCountLensStateToAPI)(options);
  }
  if ((0, _utils.isLensStateColumnOfType)('average', options) || (0, _utils.isLensStateColumnOfType)('median', options) || (0, _utils.isLensStateColumnOfType)('min', options) || (0, _utils.isLensStateColumnOfType)('max', options) || (0, _utils.isLensStateColumnOfType)('standard_deviation', options)) {
    return (0, _max_min_avg_sum.fromBasicMetricLensStateToAPI)(options);
  }
  if ((0, _utils.isLensStateColumnOfType)('sum', options)) {
    return (0, _max_min_avg_sum.fromSumMetricLensStateToAPI)(options);
  }
  if ((0, _utils.isLensStateColumnOfType)('static_value', options)) {
    return (0, _static_value.fromStaticValueLensStateToAPI)(options);
  }
  if ((0, _utils.isLensStateColumnOfType)('formula', options)) {
    return (0, _formula.fromFormulaLensStateToAPI)(options);
  }
  if ((0, _utils.isLensStateColumnOfType)('last_value', options)) {
    return (0, _last_value.fromLastValueLensStateToAPI)(options);
  }
  if ((0, _utils.isLensStateColumnOfType)('percentile', options)) {
    return (0, _percentile.fromPercentileLensStateToAPI)(options);
  }
  if ((0, _utils.isLensStateColumnOfType)('percentile_rank', options)) {
    return (0, _percentile_ranks.fromPercentileRankLensStateToAPI)(options);
  }
  if ((0, _utils.isLensStateColumnOfType)('moving_average', options)) {
    const refColumn = getMetricReferableApiColumnFromLensState(options, columns);
    return (0, _moving_average.fromMovingAverageLensStateToAPI)(options, refColumn);
  }
  if ((0, _utils.isLensStateColumnOfType)('counter_rate', options)) {
    const refColumn = getMetricReferableApiColumnFromLensState(options, columns);
    if (!(0, _utils.isAPIColumnOfType)('max', refColumn)) {
      throw new Error(`Unsupported referenced metric operation: ${options.operationType}`);
    }
    return (0, _counter_rate.fromCounterRateLensStateToAPI)(options, refColumn);
  }
  if ((0, _utils.isLensStateColumnOfType)('cumulative_sum', options)) {
    const refColumn = getMetricReferableApiColumnFromLensState(options, columns);
    if (!(0, _utils.isAPIColumnOfType)('sum', refColumn)) {
      throw new Error(`Unsupported referenced metric operation: ${options.operationType}`);
    }
    return (0, _cumulative_sum.fromCumulativeSumLensStateToAPI)(options, refColumn);
  }
  if ((0, _utils.isLensStateColumnOfType)('differences', options)) {
    const refColumn = getMetricReferableApiColumnFromLensState(options, columns);
    return (0, _differences.fromDifferencesLensStateToAPI)(options, refColumn);
  }
  throw new Error(`Unsupported metric operation`);
}