"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.staticColorSchema = exports.colorMappingSchema = exports.colorByValueSchema = exports.colorByValueAbsolute = exports.applyColorToSchema = exports.allColoringTypeSchema = void 0;
var _configSchema = require("@kbn/config-schema");
var _serializedValue = require("./serializedValue");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const colorByValueBase = _configSchema.schema.object({
  type: _configSchema.schema.literal('dynamic'),
  // Specifies that the color assignment is dynamic (by value). Possible value: 'dynamic'

  /**
   * Array of color steps defining the mapping from values to colors.
   * Each step can be:
   *   - 'from': Color applies from a specified value upwards.
   *   - 'to': Color applies up to a specified value.
   *   - 'exact': Color applies to an exact value.
   */
  steps: _configSchema.schema.arrayOf(_configSchema.schema.oneOf([_configSchema.schema.object({
    /**
     * Step type indicating the color applies from a specific value upwards.
     * Possible value: 'from'
     */
    type: _configSchema.schema.literal('from'),
    /**
     * The value from which this color applies (inclusive).
     */
    from: _configSchema.schema.number({
      meta: {
        description: 'The value from which this color applies (inclusive).'
      }
    }),
    /**
     * The color to use for this step.
     */
    color: _configSchema.schema.string({
      meta: {
        description: 'The color to use for this step.'
      }
    })
  }), _configSchema.schema.object({
    /**
     * Step type indicating the color applies up to a specific value.
     * Possible value: 'to'
     */
    type: _configSchema.schema.literal('to'),
    /**
     * The value up to which this color applies (inclusive).
     */
    to: _configSchema.schema.number({
      meta: {
        description: 'The value up to which this color applies (inclusive).'
      }
    }),
    /**
     * The color to use for this step.
     */
    color: _configSchema.schema.string({
      meta: {
        description: 'The color to use for this step.'
      }
    })
  }), _configSchema.schema.object({
    type: _configSchema.schema.literal('exact'),
    // Step type indicating the color applies to an exact value. Possible value: 'exact'
    /**
     * The exact value to which this color applies.
     */
    value: _configSchema.schema.number({
      meta: {
        description: 'The exact value to which this color applies.'
      }
    }),
    /**
     * The color to use for this exact value.
     */
    color: _configSchema.schema.string({
      meta: {
        description: 'The color to use for this exact value.'
      }
    })
  })]), {
    validate(steps) {
      if (steps.some(step => step.type === 'from') && steps.findIndex(step => step.type === 'from') !== 0) {
        return 'The "from" step must be the first step in the array.';
      }
      if (steps.some(step => step.type === 'to') && steps.findIndex(step => step.type === 'to') !== steps.length - 1) {
        return 'The "to" step must be the last step in the array.';
      }
      return undefined;
    }
  })
});
const colorByValueAbsolute = exports.colorByValueAbsolute = _configSchema.schema.allOf([colorByValueBase, _configSchema.schema.object({
  range: _configSchema.schema.literal('absolute')
})]);
const colorByValueSchema = exports.colorByValueSchema = _configSchema.schema.oneOf([colorByValueAbsolute, _configSchema.schema.allOf([colorByValueBase, _configSchema.schema.object({
  /**
   * The minimum value for the color range. Used as the lower bound for value-based color assignment.
   */
  min: _configSchema.schema.number({
    meta: {
      description: 'The minimum value for the color range. Used as the lower bound for value-based color assignment.'
    }
  }),
  /**
   * The maximum value for the color range. Used as the upper bound for value-based color assignment.
   */
  max: _configSchema.schema.number({
    meta: {
      description: 'The maximum value for the color range. Used as the upper bound for value-based color assignment.'
    }
  }),
  /**
   * Determines whether the range is interpreted as absolute or as a percentage of the data.
   * Possible values: 'absolute', 'percentage'
   */
  range: _configSchema.schema.literal('percentage') // Range is interpreted as percentage values. Possible value: 'percentage'
})])]);
const staticColorSchema = exports.staticColorSchema = _configSchema.schema.object({
  type: _configSchema.schema.literal('static'),
  // Specifies that the color assignment is static (single color for all values). Possible value: 'static'
  /**
   * The static color to be used for all values.
   */
  color: _configSchema.schema.string({
    meta: {
      description: 'The static color to be used for all values.'
    }
  })
});
const colorFromPaletteSchema = _configSchema.schema.object({
  type: _configSchema.schema.literal('from_palette'),
  index: _configSchema.schema.number({
    meta: {
      description: 'The index of the color in the palette.'
    }
  }),
  palette: _configSchema.schema.maybe(_configSchema.schema.string({
    meta: {
      description: 'The palette name to use.'
    }
  }))
});
const colorCodeSchema = _configSchema.schema.object({
  type: _configSchema.schema.literal('colorCode'),
  value: _configSchema.schema.string({
    meta: {
      description: 'The static color value to use.'
    }
  })
});
const colorDefSchema = _configSchema.schema.oneOf([colorFromPaletteSchema, colorCodeSchema]);
const categoricalColorMappingSchema = _configSchema.schema.object({
  mode: _configSchema.schema.literal('categorical'),
  palette: _configSchema.schema.string({
    meta: {
      description: 'The palette name to use for color assignment.'
    }
  }),
  mapping: _configSchema.schema.arrayOf(_configSchema.schema.object({
    values: _configSchema.schema.arrayOf(_serializedValue.serializedValueSchema),
    color: colorDefSchema
  })),
  unassignedColor: _configSchema.schema.maybe(colorCodeSchema)
});
const gradientColorMappingSchema = _configSchema.schema.object({
  mode: _configSchema.schema.literal('gradient'),
  palette: _configSchema.schema.string({
    meta: {
      description: 'The palette name to use for color assignment.'
    }
  }),
  mapping: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.object({
    values: _configSchema.schema.arrayOf(_serializedValue.serializedValueSchema)
  }))),
  gradient: _configSchema.schema.maybe(_configSchema.schema.arrayOf(colorDefSchema)),
  unassignedColor: _configSchema.schema.maybe(colorCodeSchema)
});
const colorMappingSchema = exports.colorMappingSchema = _configSchema.schema.oneOf([
/**
 * Categorical color mapping: assigns colors from a palette to specific values.
 */
categoricalColorMappingSchema,
/**
 * Gradient color mapping: assigns a gradient of colors to a range of values.
 */
gradientColorMappingSchema]);
const allColoringTypeSchema = exports.allColoringTypeSchema = _configSchema.schema.oneOf([colorByValueSchema, staticColorSchema, colorMappingSchema]);
/**
 * Schema for where to apply the color (to value or background).
 */
const applyColorToSchema = exports.applyColorToSchema = _configSchema.schema.oneOf([_configSchema.schema.literal('value'), _configSchema.schema.literal('background')], {
  meta: {
    description: 'Where to apply the color'
  }
});