"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getProcessList = void 0;
var _common = require("@kbn/metrics-data-access-plugin/common");
var _constants = require("../../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getEcsProcessList = async (infraMetricsClient, {
  hostTerm,
  to,
  sortBy,
  searchFilter
}) => {
  var _response$aggregation, _response$aggregation2, _response$aggregation3;
  const mandatoryFieldsFilter = _constants.MANDATORY_PROCESS_FIELDS_ECS.map(field => ({
    exists: {
      field
    }
  }));
  const filter = searchFilter ? searchFilter : [{
    match_all: {}
  }];
  const response = await infraMetricsClient.search({
    track_total_hits: true,
    size: 0,
    query: {
      bool: {
        filter: [{
          range: {
            [_constants.TIMESTAMP_FIELD]: {
              gte: to - 60 * 1000,
              // 1 minute
              lte: to,
              format: 'epoch_millis'
            }
          }
        }, {
          term: hostTerm
        }]
      }
    },
    aggs: {
      summaryEvent: {
        filter: {
          term: {
            'event.dataset': 'system.process.summary'
          }
        },
        aggs: {
          summary: {
            top_hits: {
              size: 1,
              sort: [{
                [_constants.TIMESTAMP_FIELD]: {
                  order: 'desc'
                }
              }],
              _source: ['system.process.summary']
            }
          }
        }
      },
      processes: {
        filter: {
          bool: {
            must: [...filter, ...mandatoryFieldsFilter]
          }
        },
        aggs: {
          filteredProcs: {
            terms: {
              field: _constants.PROCESS_COMMANDLINE_FIELD,
              size: _constants.TOP_N,
              order: {
                [sortBy.name]: sortBy.isAscending ? 'asc' : 'desc'
              }
            },
            aggs: {
              cpu: {
                avg: {
                  field: 'system.process.cpu.total.pct'
                }
              },
              memory: {
                avg: {
                  field: 'system.process.memory.rss.pct'
                }
              },
              startTime: {
                max: {
                  field: 'system.process.cpu.start_time'
                }
              },
              meta: {
                top_hits: {
                  size: 1,
                  sort: [{
                    [_constants.TIMESTAMP_FIELD]: {
                      order: 'desc'
                    }
                  }],
                  _source: ['system.process.state', 'user.name', 'process.pid', 'process.command_line']
                }
              }
            }
          }
        }
      }
    }
  });
  const {
    buckets: processListBuckets
  } = response.aggregations.processes.filteredProcs;
  const processList = processListBuckets.map(bucket => {
    var _bucket$cpu$value, _bucket$memory$value;
    const meta = bucket.meta.hits.hits[0]._source;
    return {
      cpu: (_bucket$cpu$value = bucket.cpu.value) !== null && _bucket$cpu$value !== void 0 ? _bucket$cpu$value : null,
      memory: (_bucket$memory$value = bucket.memory.value) !== null && _bucket$memory$value !== void 0 ? _bucket$memory$value : null,
      startTime: bucket.startTime.value,
      pid: meta.process.pid,
      state: meta.system.process.state,
      user: meta.user.name,
      command: bucket.key
    };
  });
  const summary = (_response$aggregation = response.aggregations) !== null && _response$aggregation !== void 0 && (_response$aggregation2 = _response$aggregation.summaryEvent.summary.hits.hits) !== null && _response$aggregation2 !== void 0 && _response$aggregation2.length ? ((_response$aggregation3 = response.aggregations) === null || _response$aggregation3 === void 0 ? void 0 : _response$aggregation3.summaryEvent.summary.hits.hits[0]._source).system.process.summary : {};
  return {
    processList,
    summary
  };
};
const getSemConvProcessList = async (infraMetricsClient, {
  hostTerm,
  to,
  sortBy,
  searchFilter
}) => {
  var _response$aggregation4;
  const mandatoryFieldsFilter = _constants.MANDATORY_PROCESS_FIELDS_SEMCONV.map(field => ({
    exists: {
      field
    }
  }));
  const filter = searchFilter ? searchFilter : [{
    match_all: {}
  }];
  const response = await infraMetricsClient.search({
    track_total_hits: true,
    size: 0,
    query: {
      bool: {
        filter: [{
          range: {
            [_constants.TIMESTAMP_FIELD]: {
              gte: to - 60 * 1000,
              // 1 minute
              lte: to,
              format: 'epoch_millis'
            }
          }
        }, {
          term: hostTerm
        }]
      }
    },
    aggs: {
      processes: {
        filter: {
          bool: {
            must: [...filter, ...mandatoryFieldsFilter]
          }
        },
        aggs: {
          filteredProcs: {
            terms: {
              field: _constants.PROCESS_COMMANDLINE_FIELD,
              size: _constants.TOP_N,
              order: {
                [sortBy.name]: sortBy.isAscending ? 'asc' : 'desc'
              }
            },
            aggs: {
              // Dual CPU aggregation approach to handle sorting vs display accuracy

              // PROBLEM: Pipeline aggregations (like sum_bucket) cannot be used for sorting in ES
              // SOLUTION: Use two separate CPU aggregations:

              // 1. 'cpu' - Simple average for sorting
              //    Used in the 'order' clause above for sorting buckets

              // 2. 'cpu_total' - Complex state-based aggregation for accurate display values
              //    Aggregates CPU utilization across all process states, then sums them up
              //    More accurate but cannot be used for sorting (pipeline aggregation)

              // This ensures sorting works correctly while maintaining accurate CPU calculations
              // the difference between simple average and state-aggregated CPU is likely minimal
              cpu: {
                avg: {
                  field: 'process.cpu.utilization'
                }
              },
              cpu_by_state: {
                terms: {
                  field: 'attributes.state',
                  size: 20
                },
                aggs: {
                  avg_cpu: {
                    avg: {
                      field: 'process.cpu.utilization'
                    }
                  }
                }
              },
              cpu_total: {
                sum_bucket: {
                  buckets_path: 'cpu_by_state>avg_cpu'
                }
              },
              memory: {
                avg: {
                  field: 'process.memory.utilization'
                }
              },
              startTime: {
                max: {
                  field: 'start_timestamp'
                }
              },
              meta: {
                top_hits: {
                  size: 1,
                  sort: [{
                    [_constants.TIMESTAMP_FIELD]: {
                      order: 'desc'
                    }
                  }],
                  fields: ['process.pid', 'process.command_line', 'process.owner']
                }
              }
            }
          }
        }
      }
    }
  });
  const processList = ((_response$aggregation4 = response.aggregations) === null || _response$aggregation4 === void 0 ? void 0 : _response$aggregation4.processes.filteredProcs.buckets.map(bucket => {
    const meta = bucket.meta.hits.hits[0].fields;
    return {
      cpu: bucket.cpu_total.value,
      memory: bucket.memory.value !== null ? bucket.memory.value / 100 : null,
      startTime: bucket.startTime.value,
      pid: meta['process.pid'][0],
      state: '',
      // Not available in SEMCONV
      user: meta['process.owner'][0],
      command: bucket.key
    };
  })) || [];
  return {
    processList,
    summary: {}
  };
};
const getProcessList = async (infraMetricsClient, {
  hostTerm,
  to,
  sortBy,
  searchFilter,
  schema,
  sourceId
}) => {
  const detectedSchema = schema || _common.DataSchemaFormatEnum.ECS;
  if (detectedSchema === _common.DataSchemaFormatEnum.SEMCONV) {
    return await getSemConvProcessList(infraMetricsClient, {
      hostTerm,
      to,
      sortBy,
      searchFilter,
      sourceId
    });
  } else {
    return await getEcsProcessList(infraMetricsClient, {
      hostTerm,
      to,
      sortBy,
      searchFilter,
      sourceId
    });
  }
};
exports.getProcessList = getProcessList;