"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.mergeChunks = void 0;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Merges tool calls from chunks into a single array of tool calls.
 *
 * Merge logic:
 * - `toolCallId` and `function.name` appear in the first chunk and should be set once
 * - `function.arguments` is streamed across multiple chunks and should be accumulated
 * - When `toolCallId` is empty, it's a continuation chunk for the most recent tool call at that index
 */
const mergeToolCalls = chunks => {
  // Map to store tool calls by their unique identifier
  const toolCallsMap = new Map();
  // Map to track which index corresponds to which toolCallId
  const indexToToolCallId = new Map();
  for (const chunk of chunks) {
    var _chunk$tool_calls;
    (_chunk$tool_calls = chunk.tool_calls) === null || _chunk$tool_calls === void 0 ? void 0 : _chunk$tool_calls.forEach(toolCall => {
      // Track toolCallId for continuation chunks
      if (toolCall.toolCallId) {
        indexToToolCallId.set(toolCall.index, toolCall.toolCallId);
      }
      const key = toolCall.toolCallId || indexToToolCallId.get(toolCall.index);
      if (!key) {
        throw new Error(`Tool call key is missing for index ${toolCall.index} in chunk ${JSON.stringify(chunk)}`);
      }
      const existingToolCall = toolCallsMap.get(key);
      const updatedToolCall = {
        function: {
          name: toolCall.function.name || (existingToolCall === null || existingToolCall === void 0 ? void 0 : existingToolCall.function.name) || '',
          arguments: ((existingToolCall === null || existingToolCall === void 0 ? void 0 : existingToolCall.function.arguments) || '') + toolCall.function.arguments
        },
        toolCallId: toolCall.toolCallId || (existingToolCall === null || existingToolCall === void 0 ? void 0 : existingToolCall.toolCallId) || ''
      };
      toolCallsMap.set(key, updatedToolCall);
    });
  }
  return Array.from(toolCallsMap.values()).map(call => {
    if (call.function.arguments === '') {
      return {
        ...call,
        function: {
          ...call.function,
          arguments: '{}'
        }
      };
    }
    return call;
  });
};

/**
 * Merges chunks into a message, concatenating the content and tool calls.
 */
const mergeChunks = chunks => {
  const message = chunks.reduce((prev, chunk) => {
    var _chunk$content;
    prev.content += (_chunk$content = chunk.content) !== null && _chunk$content !== void 0 ? _chunk$content : '';
    if (chunk.refusal) {
      prev.refusal = chunk.refusal;
    }
    return prev;
  }, {
    content: '',
    tool_calls: []
  });
  message.tool_calls = mergeToolCalls(chunks);
  return message;
};
exports.mergeChunks = mergeChunks;