"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.processCompletionChunks = processCompletionChunks;
exports.processConverseCompletionChunks = processConverseCompletionChunks;
var _rxjs = require("rxjs");
var _inferenceCommon = require("@kbn/inference-common");
var _mlIsPopulatedObject = require("@kbn/ml-is-populated-object");
var _types = require("./types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function processCompletionChunks(model) {
  return source => new _rxjs.Observable(subscriber => {
    function handleNext(chunkBody) {
      if (isTokenCountCompletionChunk(chunkBody)) {
        return emitTokenCountEvent(subscriber, chunkBody, model);
      }
      let completionChunk = '';
      let toolCallChunk;
      switch (chunkBody.type) {
        case 'content_block_start':
          if (chunkBody.content_block.type === 'text') {
            completionChunk = chunkBody.content_block.text || '';
          } else if (chunkBody.content_block.type === 'tool_use') {
            toolCallChunk = {
              index: chunkBody.index,
              toolCallId: chunkBody.content_block.id,
              function: {
                name: chunkBody.content_block.name,
                // the API returns '{}' here, which can't be merged with the deltas...
                arguments: ''
              }
            };
          }
          break;
        case 'content_block_delta':
          if (chunkBody.delta.type === 'text_delta') {
            completionChunk = chunkBody.delta.text || '';
          } else if (chunkBody.delta.type === 'input_json_delta') {
            toolCallChunk = {
              index: chunkBody.index,
              toolCallId: '',
              function: {
                name: '',
                arguments: chunkBody.delta.partial_json
              }
            };
          }
          break;
        case 'message_delta':
          completionChunk = chunkBody.delta.stop_sequence || '';
          break;
        default:
          break;
      }
      if (completionChunk || toolCallChunk) {
        subscriber.next({
          type: _inferenceCommon.ChatCompletionEventType.ChatCompletionChunk,
          content: completionChunk,
          tool_calls: toolCallChunk ? [toolCallChunk] : []
        });
      }
    }
    source.subscribe({
      next: value => {
        try {
          handleNext(value);
        } catch (error) {
          subscriber.error(error);
        }
      },
      error: err => {
        subscriber.error(err);
      },
      complete: () => {
        subscriber.complete();
      }
    });
  });
}
function isOfType(chunkBody, type, expectedType) {
  return type === expectedType;
}
function processConverseCompletionChunks(model) {
  return source => new _rxjs.Observable(subscriber => {
    function handleNext({
      type,
      body: chunkBody
    }) {
      if (type === 'metadata' && isConverseStreamMetadataEvent(chunkBody)) {
        return emitTokenCountEvent(subscriber, chunkBody, model);
      }
      let completionChunk = '';
      let toolCallChunk;
      if (isOfType(chunkBody, type, 'contentBlockStart')) {
        var _chunkBody$start;
        if ((_chunkBody$start = chunkBody.start) !== null && _chunkBody$start !== void 0 && _chunkBody$start.toolUse && chunkBody.start.toolUse.toolUseId && chunkBody.start.toolUse.name && chunkBody.contentBlockIndex !== undefined) {
          toolCallChunk = {
            index: chunkBody.contentBlockIndex,
            toolCallId: chunkBody.start.toolUse.toolUseId,
            function: {
              name: chunkBody.start.toolUse.name,
              // the API returns '{}' here, which can't be merged with the deltas...
              arguments: ''
            }
          };
        }
      }
      if (isOfType(chunkBody, type, 'contentBlockDelta')) {
        var _chunkBody$delta, _chunkBody$delta2;
        if ((_chunkBody$delta = chunkBody.delta) !== null && _chunkBody$delta !== void 0 && _chunkBody$delta.text) {
          completionChunk = chunkBody.delta.text || '';
        } else if ((_chunkBody$delta2 = chunkBody.delta) !== null && _chunkBody$delta2 !== void 0 && _chunkBody$delta2.toolUse && chunkBody.delta.toolUse.input && chunkBody.contentBlockIndex !== undefined) {
          toolCallChunk = {
            index: chunkBody.contentBlockIndex,
            toolCallId: '',
            function: {
              name: '',
              arguments: chunkBody.delta.toolUse.input
            }
          };
        }
      }
      if (completionChunk || toolCallChunk) {
        subscriber.next({
          type: _inferenceCommon.ChatCompletionEventType.ChatCompletionChunk,
          content: completionChunk,
          tool_calls: toolCallChunk ? [toolCallChunk] : []
        });
      }
    }
    source.subscribe({
      next: value => {
        try {
          handleNext(value);
        } catch (error) {
          subscriber.error(error);
        }
      },
      error: err => {
        subscriber.error(err);
      },
      complete: () => {
        subscriber.complete();
      }
    });
  });
}

/**
 *
 * @param value For BedRock Invoke
 * @returns
 */
function isTokenCountCompletionChunk(value) {
  return value.type === 'message_stop' && 'amazon-bedrock-invocationMetrics' in value;
}

/**
 * Check if the object is a ConverseStreamMetadataEvent from BedRock Converse API
 * @param arg response from BedRock Converse API
 * @returns
 */
const isConverseStreamMetadataEvent = arg => {
  return (0, _mlIsPopulatedObject.isPopulatedObject)(arg, ['usage', 'metrics']) && (0, _mlIsPopulatedObject.isPopulatedObject)(arg.usage, ['inputTokens', 'outputTokens']);
};
function emitTokenCountEvent(subscriber, chunk, model) {
  let inputTokenCount = 0;
  let outputTokenCount = 0;
  // Response from BedRock Invoke API
  if ((0, _types.isMessageStopChunk)(chunk)) {
    inputTokenCount = chunk['amazon-bedrock-invocationMetrics'].inputTokenCount;
    outputTokenCount = chunk['amazon-bedrock-invocationMetrics'].outputTokenCount;
  }
  // Response from BedRock Converse API
  if (isConverseStreamMetadataEvent(chunk)) {
    var _chunk$usage$inputTok, _chunk$usage$outputTo;
    inputTokenCount = (_chunk$usage$inputTok = chunk.usage.inputTokens) !== null && _chunk$usage$inputTok !== void 0 ? _chunk$usage$inputTok : 0;
    outputTokenCount = (_chunk$usage$outputTo = chunk.usage.outputTokens) !== null && _chunk$usage$outputTo !== void 0 ? _chunk$usage$outputTo : 0;
  }
  subscriber.next({
    type: _inferenceCommon.ChatCompletionEventType.ChatCompletionTokenCount,
    tokens: {
      completion: outputTokenCount,
      prompt: inputTokenCount,
      total: inputTokenCount + outputTokenCount
    },
    ...(model ? {
      model
    } : {})
  });
}