"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.correctLikeWildcards = void 0;
var _esqlAst = require("@kbn/esql-ast");
var _typeguards = require("../typeguards");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Correct wrong LIKE wildcard mistakes.
 * The LLM can make mistake and use SQL wildcards for LIKE operators.
 *
 * E.g.
 * `column LIKE "ba_"` => `column LIKE "ba?"`
 * `column LIKE "ba%"` => `column LIKE "ba*"`
 */
const correctLikeWildcards = query => {
  const corrections = [];
  _esqlAst.Walker.walk(query, {
    visitFunction: node => {
      if ((0, _typeguards.isLikeOperatorNode)(node)) {
        corrections.push(...checkLikeNode(node));
      }
    }
  });
  return corrections;
};
exports.correctLikeWildcards = correctLikeWildcards;
function checkLikeNode(node) {
  if (node.args.length !== 2 || !(0, _typeguards.isStringLiteralNode)(node.args[1])) {
    return [];
  }
  const likeExpression = node.args[1];
  const initialValue = likeExpression.value;
  likeExpression.value = likeExpression.value.replaceAll(/(?<!\\)%/g, '*').replaceAll(/(?<!\\)_/g, '?');
  likeExpression.valueUnquoted = likeExpression.valueUnquoted.replaceAll(/(?<!\\)%/g, '*').replaceAll(/(?<!\\)_/g, '?');
  if (likeExpression.value !== initialValue) {
    likeExpression.name = likeExpression.value;
    const correction = {
      type: 'wrong_like_wildcard',
      node,
      description: `Replaced wrong like wildcard in LIKE operator at position ${node.location.min}`
    };
    return [correction];
  }
  return [];
}