"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useCompatibleInferenceEndpoints = void 0;
var _react = require("react");
var _inferenceCommon = require("@kbn/inference-common");
var _use_license = require("./use_license");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const COMPATIBLE_TASK_TYPES = ['text_embedding', 'sparse_embedding'];
const LICENSE_TIER_ENTERPRISE = 'enterprise';
const LICENSE_TIER_PLATINUM = 'platinum';
const INFERENCE_ENDPOINT_LICENSE_MAP = {
  [_inferenceCommon.defaultInferenceEndpoints.ELSER]: LICENSE_TIER_PLATINUM,
  [_inferenceCommon.defaultInferenceEndpoints.ELSER_IN_EIS_INFERENCE_ID]: LICENSE_TIER_ENTERPRISE
};
/**
 * Transforms the inference endpoints into a format that can be used to build the selectable options for the inference endpoint dropdown.
 *
 * Remains in loading state until the endpoints and license are loaded.
 */
const useCompatibleInferenceEndpoints = (endpoints, endpointsLoading) => {
  const {
    isLoading: licenseLoading,
    isAtLeast
  } = (0, _use_license.useLicense)();
  const compatibleEndpoints = (0, _react.useMemo)(() => {
    if (endpointsLoading || licenseLoading) {
      return;
    }
    let defaultInferenceId;
    const endpointDefinitions = [];
    endpoints === null || endpoints === void 0 ? void 0 : endpoints.forEach(endpoint => {
      // Skip incompatible endpoints
      if (!COMPATIBLE_TASK_TYPES.includes(endpoint.task_type)) {
        return;
      }
      const isElserInEis = endpoint.inference_id === _inferenceCommon.defaultInferenceEndpoints.ELSER_IN_EIS_INFERENCE_ID;
      const requiredLicense = INFERENCE_ENDPOINT_LICENSE_MAP[endpoint.inference_id];
      // If no license requirement is specified, assume access is granted.
      const accessible = requiredLicense ? isAtLeast(requiredLicense) : true;
      if (accessible) {
        if (isElserInEis) {
          // Prioritize elser in eis endpoint as default.
          defaultInferenceId = endpoint.inference_id;
        } else if (!defaultInferenceId) {
          // Otherwise use the first accessible endpoint as default.
          defaultInferenceId = endpoint.inference_id;
        }
      }
      endpointDefinitions.push({
        inference_id: endpoint.inference_id,
        requiredLicense,
        accessible
      });
    });
    return {
      defaultInferenceId,
      endpointDefinitions
    };
  }, [endpointsLoading, licenseLoading, endpoints, isAtLeast]);
  return {
    compatibleEndpoints,
    isLoading: !compatibleEndpoints
  };
};
exports.useCompatibleInferenceEndpoints = useCompatibleInferenceEndpoints;