"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getMaxNestedDepth = exports.getFieldsMatchingFilterFromState = exports.getFieldsFromState = exports.getFieldMeta = exports.getFieldConfig = exports.getFieldByPathName = exports.getFieldAncestors = exports.getChildFieldsName = exports.getAllFieldTypesFromState = exports.getAllDescendantAliases = exports.getAllChildFields = exports.filterTypesForNonRootFields = exports.filterTypesForMultiField = exports.deNormalizeRuntimeFields = exports.deNormalize = exports.canUseMappingsEditor = exports.buildFieldTreeFromIds = void 0;
exports.getStateWithCopyToFields = getStateWithCopyToFields;
exports.getUniqueId = exports.getTypeMetaFromSource = exports.getTypeLabelFromField = void 0;
exports.hasElserOnMlNodeSemanticTextField = hasElserOnMlNodeSemanticTextField;
exports.hasSemanticTextField = hasSemanticTextField;
exports.isElserOnMlNodeSemanticField = void 0;
exports.isLocalModel = isLocalModel;
exports.isSemanticTextField = isSemanticTextField;
exports.updateFieldsPathAfterFieldNameChange = exports.stripUndefinedValues = exports.shouldDeleteChildFieldsAfterTypeChange = exports.prepareFieldsForEisUpdate = exports.normalizeRuntimeFields = exports.normalize = void 0;
var _uuid = require("uuid");
var _lodash = require("lodash");
var _inferenceCommon = require("@kbn/inference-common");
var _constants = require("../constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getUniqueId = () => (0, _uuid.v4)();
exports.getUniqueId = getUniqueId;
const fieldsWithoutMultiFields = [
// @ts-expect-error aggregate_metric_double is not yet supported by the editor
'aggregate_metric_double', 'constant_keyword', 'flattened', 'geo_shape', 'join', 'percolator', 'point', 'shape'];
const getChildFieldsName = dataType => {
  if (fieldsWithoutMultiFields.includes(dataType)) {
    return undefined;
  } else if (dataType === 'object' || dataType === 'nested') {
    return 'properties';
  }
  return 'fields';
};
exports.getChildFieldsName = getChildFieldsName;
const getFieldMeta = (field, isMultiField) => {
  const childFieldsName = getChildFieldsName(field.type);
  const canHaveChildFields = isMultiField ? false : childFieldsName === 'properties';
  const hasChildFields = isMultiField ? false : canHaveChildFields && Boolean(field[childFieldsName]) && Object.keys(field[childFieldsName]).length > 0;
  const canHaveMultiFields = isMultiField ? false : childFieldsName === 'fields';
  const hasMultiFields = isMultiField ? false : canHaveMultiFields && Boolean(field[childFieldsName]) && Object.keys(field[childFieldsName]).length > 0;
  return {
    childFieldsName,
    canHaveChildFields,
    hasChildFields,
    canHaveMultiFields,
    hasMultiFields,
    isExpanded: false
  };
};
exports.getFieldMeta = getFieldMeta;
const getTypeLabel = type => {
  return type && _constants.TYPE_DEFINITION[type] ? _constants.TYPE_DEFINITION[type].label : `${_constants.TYPE_DEFINITION.other.label}: ${type}`;
};
const getTypeLabelFromField = field => {
  const {
    type
  } = field;
  const typeLabel = getTypeLabel(type);
  return typeLabel;
};
exports.getTypeLabelFromField = getTypeLabelFromField;
const getFieldConfig = (param, prop) => {
  var _PARAMETERS_DEFINITIO;
  if (prop !== undefined) {
    var _props$prop;
    if (!_constants.PARAMETERS_DEFINITION[param].props || !_constants.PARAMETERS_DEFINITION[param].props[prop]) {
      throw new Error(`No field config found for prop "${prop}" on param "${param}" `);
    }
    return ((_props$prop = _constants.PARAMETERS_DEFINITION[param].props[prop]) === null || _props$prop === void 0 ? void 0 : _props$prop.fieldConfig) || {};
  }
  return ((_PARAMETERS_DEFINITIO = _constants.PARAMETERS_DEFINITION[param]) === null || _PARAMETERS_DEFINITIO === void 0 ? void 0 : _PARAMETERS_DEFINITIO.fieldConfig) || {};
};

/**
 * For "alias" field types, we work internaly by "id" references. When we normalize the fields, we need to
 * replace the actual "path" parameter with the field (internal) `id` the alias points to.
 * This method takes care of doing just that.
 *
 * @param byId The fields map by id
 */
exports.getFieldConfig = getFieldConfig;
const replaceAliasPathByAliasId = byId => {
  const aliases = {};
  Object.entries(byId).forEach(([id, field]) => {
    if (field.source.type === 'alias') {
      const aliasTargetField = Object.values(byId).find(_field => _field.path.join('.') === field.source.path);
      if (aliasTargetField) {
        // we set the path to the aliasTargetField "id"
        field.source.path = aliasTargetField.id;

        // We add the alias field to our "aliases" map
        aliases[aliasTargetField.id] = aliases[aliasTargetField.id] || [];
        aliases[aliasTargetField.id].push(id);
      }
    }
  });
  return {
    aliases,
    byId
  };
};
const getMainTypeFromSubType = subType => {
  var _SUB_TYPE_MAP_TO_MAIN;
  return (_SUB_TYPE_MAP_TO_MAIN = _constants.SUB_TYPE_MAP_TO_MAIN[subType]) !== null && _SUB_TYPE_MAP_TO_MAIN !== void 0 ? _SUB_TYPE_MAP_TO_MAIN : 'other';
};

/**
 * Read the field source type and decide if it is a SubType of a MainType
 * A SubType is for example the "float" datatype. It is the SubType of the "numeric" MainType
 *
 * @param sourceType The type declared on the mappings field
 */
const getTypeMetaFromSource = sourceType => {
  if (!_constants.MAIN_DATA_TYPE_DEFINITION[sourceType]) {
    // If the sourceType provided if **not** one of the MainType, it is probably a SubType type
    const mainType = getMainTypeFromSubType(sourceType);
    if (!mainType) {
      throw new Error(`Property type "${sourceType}" not recognized and no subType was found for it.`);
    }
    return {
      mainType,
      subType: sourceType
    };
  }
  return {
    mainType: sourceType
  };
};

/**
 * In order to better work with the recursive pattern of the mappings `properties`, this method flatten the fields
 * to a `byId` object where the key is the **path** to the field and the value is a `NormalizedField`.
 * The `NormalizedField` contains the field data under `source` and meta information about the capability of the field.
 *
 * @example

// original
{
  myObject: {
    type: 'object',
    properties: {
      name: {
        type: 'text'
      }
    }
  }
}

// normalized
{
  rootLevelFields: ['_uniqueId123'],
  byId: {
    '_uniqueId123': {
      source: { type: 'object' },
      id: '_uniqueId123',
      parentId: undefined,
      hasChildFields: true,
      childFieldsName: 'properties', // "object" type have their child fields under "properties"
      canHaveChildFields: true,
      childFields: ['_uniqueId456'],
    },
    '_uniqueId456': {
      source: { type: 'text' },
      id: '_uniqueId456',
      parentId: '_uniqueId123',
      hasChildFields: false,
      childFieldsName: 'fields', // "text" type have their child fields under "fields"
      canHaveChildFields: true,
      childFields: undefined,
    },
  },
}
 *
 * @param fieldsToNormalize The "properties" object from the mappings (or "fields" object for `text` and `keyword` types)
 */
exports.getTypeMetaFromSource = getTypeMetaFromSource;
const normalize = (fieldsToNormalize = {}) => {
  let maxNestedDepth = 0;
  const normalizeFields = (props, to, paths, arrayToKeepRef, nestedDepth, isMultiField = false, parentId) => Object.entries(props).sort(([a], [b]) => a > b ? 1 : a < b ? -1 : 0).reduce((acc, [propName, value]) => {
    const id = getUniqueId();
    arrayToKeepRef.push(id);
    const field = {
      name: propName,
      ...value
    };

    // In some cases for object, the "type" is not defined but the field
    // has properties defined. The mappings editor requires a "type" to be defined
    // so we add it here.
    if (field.type === undefined && field.properties !== undefined) {
      field.type = 'object';
    }
    const meta = getFieldMeta(field, isMultiField);
    const {
      childFieldsName,
      hasChildFields,
      hasMultiFields
    } = meta;
    if (hasChildFields || hasMultiFields) {
      const nextDepth = meta.canHaveChildFields || meta.canHaveMultiFields ? nestedDepth + 1 : nestedDepth;
      meta.childFields = [];
      maxNestedDepth = Math.max(maxNestedDepth, nextDepth);
      normalizeFields(field[childFieldsName], to, [...paths, propName], meta.childFields, nextDepth, meta.canHaveMultiFields, id);
    }
    const {
      properties,
      fields,
      ...rest
    } = field;
    const normalizedField = {
      id,
      parentId,
      nestedDepth,
      isMultiField,
      path: paths.length ? [...paths, propName] : [propName],
      source: rest,
      ...meta
    };
    acc[id] = normalizedField;
    return acc;
  }, to);
  const rootLevelFields = [];
  const {
    byId,
    aliases
  } = replaceAliasPathByAliasId(normalizeFields(fieldsToNormalize, {}, [], rootLevelFields, 0));
  return {
    byId,
    aliases,
    rootLevelFields,
    maxNestedDepth
  };
};

/**
 * The alias "path" value internally point to a field "id" (not its path). When we deNormalize the fields,
 * we need to replace the target field "id" by its actual "path", making sure to not mutate our state "fields" object.
 *
 * @param aliases The aliases map
 * @param byId The fields map by id
 */
exports.normalize = normalize;
const replaceAliasIdByAliasPath = (aliases, byId) => {
  const updatedById = {
    ...byId
  };
  Object.entries(aliases).forEach(([targetId, aliasesIds]) => {
    const path = updatedById[targetId] ? updatedById[targetId].path.join('.') : '';
    aliasesIds.forEach(id => {
      const aliasField = updatedById[id];
      if (!aliasField) {
        return;
      }
      const fieldWithUpdatedPath = {
        ...aliasField,
        source: {
          ...aliasField.source,
          path
        }
      };
      updatedById[id] = fieldWithUpdatedPath;
    });
  });
  return updatedById;
};
const deNormalize = ({
  rootLevelFields,
  byId,
  aliases
}) => {
  const serializedFieldsById = replaceAliasIdByAliasPath(aliases, byId);
  const deNormalizePaths = (ids, to = {}) => {
    ids.forEach(id => {
      const {
        source,
        childFields,
        childFieldsName
      } = serializedFieldsById[id];
      const {
        name,
        ...normalizedField
      } = source;
      const field = normalizedField;
      to[name] = field;
      if (childFields) {
        field[childFieldsName] = {};
        return deNormalizePaths(childFields, field[childFieldsName]);
      }
    });
    return to;
  };
  return deNormalizePaths(rootLevelFields);
};

/**
 * If we change the "name" of a field, we need to update its `path` and the
 * one of **all** of its child properties or multi-fields.
 *
 * @param field The field who's name has changed
 * @param byId The map of all the document fields
 */
exports.deNormalize = deNormalize;
const updateFieldsPathAfterFieldNameChange = (field, byId) => {
  const updatedById = {
    ...byId
  };
  const paths = field.parentId ? byId[field.parentId].path : [];
  const updateFieldPath = (_field, _paths) => {
    const {
      name
    } = _field.source;
    const path = _paths.length === 0 ? [name] : [..._paths, name];
    updatedById[_field.id] = {
      ..._field,
      path
    };
    if (_field.hasChildFields || _field.hasMultiFields) {
      _field.childFields.map(fieldId => byId[fieldId]).forEach(childField => {
        updateFieldPath(childField, [..._paths, name]);
      });
    }
  };
  updateFieldPath(field, paths);
  return {
    updatedFieldPath: updatedById[field.id].path,
    updatedById
  };
};

/**
 * Retrieve recursively all the children fields of a field
 *
 * @param field The field to return the children from
 * @param byId Map of all the document fields
 */
exports.updateFieldsPathAfterFieldNameChange = updateFieldsPathAfterFieldNameChange;
const getAllChildFields = (field, byId) => {
  const getChildFields = (_field, to = []) => {
    if (_field.hasChildFields || _field.hasMultiFields) {
      _field.childFields.map(fieldId => byId[fieldId]).forEach(childField => {
        to.push(childField);
        getChildFields(childField, to);
      });
    }
    return to;
  };
  return getChildFields(field);
};

/**
 * If we delete an object with child fields or a text/keyword with multi-field,
 * we need to know if any of its "child" fields has an `alias` that points to it.
 * This method traverse the field descendant tree and returns all the aliases found
 * on the field and its possible children.
 */
exports.getAllChildFields = getAllChildFields;
const getAllDescendantAliases = (field, fields, aliasesIds = []) => {
  const hasAliases = fields.aliases[field.id] && Boolean(fields.aliases[field.id].length);
  if (!hasAliases && !field.hasChildFields && !field.hasMultiFields) {
    return aliasesIds;
  }
  if (hasAliases) {
    fields.aliases[field.id].forEach(id => {
      aliasesIds.push(id);
    });
  }
  if (field.childFields) {
    field.childFields.forEach(id => {
      if (!fields.byId[id]) {
        return;
      }
      getAllDescendantAliases(fields.byId[id], fields, aliasesIds);
    });
  }
  return aliasesIds;
};

/**
 * Helper to retrieve a map of all the ancestors of a field
 *
 * @param fieldId The field id
 * @param byId A map of all the fields by Id
 */
exports.getAllDescendantAliases = getAllDescendantAliases;
const getFieldAncestors = (fieldId, byId) => {
  const ancestors = {};
  const currentField = byId[fieldId];
  let parent = currentField.parentId === undefined ? undefined : byId[currentField.parentId];
  while (parent) {
    ancestors[parent.id] = true;
    parent = parent.parentId === undefined ? undefined : byId[parent.parentId];
  }
  return ancestors;
};
exports.getFieldAncestors = getFieldAncestors;
const filterTypesForMultiField = options => options.filter(option => _constants.TYPE_NOT_ALLOWED_MULTIFIELD.includes(option.value) === false);
exports.filterTypesForMultiField = filterTypesForMultiField;
const filterTypesForNonRootFields = options => options.filter(option => _constants.TYPE_ONLY_ALLOWED_AT_ROOT_LEVEL.includes(option.value) === false);

/**
 * Return the max nested depth of the document fields
 *
 * @param byId Map of all the document fields
 */
exports.filterTypesForNonRootFields = filterTypesForNonRootFields;
const getMaxNestedDepth = byId => Object.values(byId).reduce((maxDepth, field) => {
  return Math.max(maxDepth, field.nestedDepth);
}, 0);

/**
 * Create a nested array of fields and its possible children
 * to render a Tree view of them.
 */
exports.getMaxNestedDepth = getMaxNestedDepth;
const buildFieldTreeFromIds = (fieldsIds, byId, render) => fieldsIds.map(id => {
  const field = byId[id];
  const children = field.childFields ? buildFieldTreeFromIds(field.childFields, byId, render) : undefined;
  return {
    label: render(field),
    children
  };
});

/**
 * When changing the type of a field, in most cases we want to delete all its child fields.
 * There are some exceptions, when changing from "text" to "keyword" as both have the same "fields" property.
 */
exports.buildFieldTreeFromIds = buildFieldTreeFromIds;
const shouldDeleteChildFieldsAfterTypeChange = (oldType, newType) => {
  if (oldType === 'text' && newType !== 'keyword') {
    return true;
  } else if (oldType === 'keyword' && newType !== 'text') {
    return true;
  } else if (oldType === 'object' && newType !== 'nested') {
    return true;
  } else if (oldType === 'nested' && newType !== 'object') {
    return true;
  }
  return false;
};
exports.shouldDeleteChildFieldsAfterTypeChange = shouldDeleteChildFieldsAfterTypeChange;
const canUseMappingsEditor = maxNestedDepth => maxNestedDepth < _constants.MAX_DEPTH_DEFAULT_EDITOR;

/**
 * This helper removes all the keys on an object with an "undefined" value.
 * To avoid sending updates from the mappings editor with this type of object:
 *
 *```
 * {
 *   "dyamic": undefined,
 *   "date_detection": undefined,
 *   "dynamic": undefined,
 *   "dynamic_date_formats": undefined,
 *   "dynamic_templates": undefined,
 *   "numeric_detection": undefined,
 *   "properties": {
 *     "title": { "type": "text" }
 *   }
 * }
 *```
 *
 * @param obj The object to retrieve the undefined values from
 * @param recursive A flag to strip recursively into children objects
 */
exports.canUseMappingsEditor = canUseMappingsEditor;
const stripUndefinedValues = (obj, recursive = true) => Object.entries(obj).reduce((acc, [key, value]) => {
  if (value === undefined) {
    return acc;
  }
  if (Array.isArray(value) || value instanceof Date || value === null) {
    return {
      ...acc,
      [key]: value
    };
  }
  return recursive && typeof value === 'object' ? {
    ...acc,
    [key]: stripUndefinedValues(value, recursive)
  } : {
    ...acc,
    [key]: value
  };
}, {});
exports.stripUndefinedValues = stripUndefinedValues;
const normalizeRuntimeFields = (fields = {}) => {
  return Object.entries(fields).reduce((acc, [name, field]) => {
    const id = getUniqueId();
    return {
      ...acc,
      [id]: {
        id,
        source: {
          name,
          ...field
        }
      }
    };
  }, {});
};
exports.normalizeRuntimeFields = normalizeRuntimeFields;
const deNormalizeRuntimeFields = fields => {
  return Object.values(fields).reduce((acc, {
    source
  }) => {
    const {
      name,
      ...rest
    } = source;
    return {
      ...acc,
      [name]: rest
    };
  }, {});
};

/**
 * get all the fields from given state which matches selected DataTypes from filter
 *
 * @param state The state that we are using depending on the context (when adding new fields, static state is used)
 * @param filteredDataTypes data types array from which fields are filtered from given state
 */
exports.deNormalizeRuntimeFields = deNormalizeRuntimeFields;
const getFieldsMatchingFilterFromState = (state, filteredDataTypes) => {
  return Object.fromEntries(Object.entries(state.fields.byId).filter(([_, fieldId]) => filteredDataTypes.includes(getTypeLabelFromField(state.fields.byId[fieldId.id].source))));
};

/** accepts Generics argument and returns value, if value is not null or undefined
 * @param value
 */
exports.getFieldsMatchingFilterFromState = getFieldsMatchingFilterFromState;
function isNotNullish(value) {
  return value !== null && value !== undefined;
}

/** returns normalized field that matches the dataTypes from the filteredDataTypes array
 * @param normalizedFields fields that we are using, depending on the context (when adding new fields, static state is used)
 * @param filteredDataTypes data types array from which fields are filtered from given state. When there are no filter selected, array would be undefined
 */
const getFieldsFromState = (normalizedFields, filteredDataTypes) => {
  const getField = fieldId => {
    if (filteredDataTypes) {
      if (filteredDataTypes.includes(getTypeLabelFromField(normalizedFields.byId[fieldId].source))) {
        return normalizedFields.byId[fieldId];
      }
    } else {
      return normalizedFields.byId[fieldId];
    }
  };
  const fields = filteredDataTypes ? Object.entries(normalizedFields.byId).map(([key, _]) => getField(key)) : normalizedFields.rootLevelFields.map(id => getField(id));
  return fields.filter(isNotNullish);
};
/**
 * returns true if given value is first occurence of array
 * useful when filtering unique values of an array
 */
exports.getFieldsFromState = getFieldsFromState;
function filterUnique(value, index, array) {
  return array.indexOf(value) === index;
}
/**
 * returns array consisting of all field types from state's fields including nested fields
 * @param fields
 */
const getallFieldsIncludingNestedFields = (fields, fieldsArray) => {
  const fieldsValue = Object.values(fields);
  for (const field of fieldsValue) {
    if (field.type) fieldsArray.push(field.type);
    if (field.fields) getallFieldsIncludingNestedFields(field.fields, fieldsArray);
    if (field.properties) getallFieldsIncludingNestedFields(field.properties, fieldsArray);
  }
  return fieldsArray;
};

/** returns all field types from the fields, including multifield and child fields
 * @param allFields fields from state
 */
const getAllFieldTypesFromState = allFields => {
  const fields = [];
  return getallFieldsIncludingNestedFields(allFields, fields).filter(filterUnique);
};
exports.getAllFieldTypesFromState = getAllFieldTypesFromState;
function isSemanticTextField(field) {
  return field.type === 'semantic_text';
}

/**
 * Returns deep copy of state with `copy_to` added to text fields that are referenced by new semantic text fields
 * @param state
 * @returns state
 */

function getStateWithCopyToFields(state) {
  // Make sure we don't accidentally modify existing state
  let updatedState = (0, _lodash.cloneDeep)(state);
  for (const field of Object.values(updatedState.fields.byId)) {
    if (field.source.type === 'semantic_text') {
      // Check fields already added to the list of to-update fields first
      // API will not accept reference_field so removing it now
      const {
        reference_field: referenceField,
        ...source
      } = field.source;
      if (typeof referenceField !== 'string') {
        // should never happen
        throw new Error('Reference field is not a string');
      }
      field.source = source;

      /*
        If no reference field is associated,
        no further processing is needed, so we can skip to the next one.
      */
      if ((0, _lodash.isEmpty)(referenceField)) {
        continue;
      }
      const existingTextField = getFieldByPathName(updatedState.fields, referenceField) || getFieldByPathName(updatedState.mappingViewFields || {
        byId: {}
      }, referenceField);
      if (existingTextField) {
        // Add copy_to to existing text field's copy_to array
        const updatedTextField = {
          ...existingTextField,
          source: {
            ...existingTextField.source,
            copy_to: existingTextField.source.copy_to ? [...(Array.isArray(existingTextField.source.copy_to) ? existingTextField.source.copy_to : [existingTextField.source.copy_to]), field.path.join('.')] : [field.path.join('.')]
          }
        };
        updatedState = {
          ...updatedState,
          fields: {
            ...updatedState.fields,
            byId: {
              ...updatedState.fields.byId,
              [existingTextField.id]: updatedTextField
            }
          }
        };
        addChildFieldsToState(updatedTextField, updatedState);
        if (existingTextField.parentId) {
          let currentField = existingTextField;
          let hasParent = true;
          while (hasParent) {
            if (!currentField.parentId) {
              // reached the top of the tree, push current field to root level fields
              updatedState.fields.rootLevelFields.push(currentField.id);
              hasParent = false;
            } else if (updatedState.fields.byId[currentField.parentId]) {
              // parent is already in state, don't need to do anything
              hasParent = false;
            } else {
              // parent is not in state yet
              updatedState.fields.byId[currentField.parentId] = updatedState.mappingViewFields.byId[currentField.parentId];
              addChildFieldsToState(updatedState.mappingViewFields.byId[currentField.parentId], updatedState);
              currentField = updatedState.fields.byId[currentField.parentId];
            }
          }
        } else {
          updatedState.fields.rootLevelFields.push(existingTextField.id);
        }
      } else {
        throw new Error(`Semantic text field ${field.path.join('.')} has invalid reference field`);
      }
    }
  }
  return updatedState;
}
function addChildFieldsToState(field, state) {
  if (!field.childFields || field.childFields.length === 0) {
    return state;
  }
  for (const childFieldId of field.childFields) {
    if (!state.fields.byId[childFieldId]) {
      state.fields.byId[childFieldId] = state.mappingViewFields.byId[childFieldId];
      state = addChildFieldsToState(state.fields.byId[childFieldId], state);
    }
  }
  return state;
}
const getFieldByPathName = (fields, name) => {
  return Object.values(fields.byId).find(field => field.path.join('.') === name);
};
exports.getFieldByPathName = getFieldByPathName;
function isLocalModel(model) {
  return ['elser', 'elasticsearch'].includes(model.service);
}
const isElserOnMlNodeSemanticField = field => field.source.inference_id === _inferenceCommon.defaultInferenceEndpoints.ELSER;
exports.isElserOnMlNodeSemanticField = isElserOnMlNodeSemanticField;
function hasElserOnMlNodeSemanticTextField(fields) {
  return Object.values(fields.byId).some(isElserOnMlNodeSemanticField);
}
function hasSemanticTextField(fields) {
  return Object.values(fields.byId).some(field => field.source.type === 'semantic_text');
}
const prepareFieldsForEisUpdate = (selectedMappings, fullNormalized) => {
  const selectedIds = selectedMappings.flatMap(item => {
    var _item$key;
    return (_item$key = item.key) !== null && _item$key !== void 0 ? _item$key : [];
  });
  const {
    byId
  } = fullNormalized;
  const resultById = {};
  const resultRootLevel = [];
  function getSelectedFieldData(id) {
    var _clonedField$source;
    // Prevent duplicate processing - if already in resultById, skip
    if (resultById[id]) {
      return;
    }
    const field = byId[id];
    if (!field) return;
    const clonedField = {
      ...field
    };

    // Only update inference_id if it exists in source
    if (((_clonedField$source = clonedField.source) === null || _clonedField$source === void 0 ? void 0 : _clonedField$source.inference_id) !== undefined) {
      clonedField.source = {
        ...clonedField.source,
        inference_id: _inferenceCommon.defaultInferenceEndpoints.ELSER_IN_EIS_INFERENCE_ID
      };
    }
    resultById[id] = clonedField;

    // Include parent if it exists and hasn't been processed yet
    if (field.parentId) {
      if (!resultById[field.parentId]) {
        getSelectedFieldData(field.parentId);
      }
    } else {
      resultRootLevel.push(id);
    }
  }
  selectedIds.forEach(id => getSelectedFieldData(id));

  // Prune childFields arrays so they only include selected field IDs
  Object.values(resultById).forEach(field => {
    if (field.childFields) {
      field.childFields = field.childFields.filter(id => !!resultById[id]);
      if (field.childFields.length === 0) {
        delete field.childFields;
      }
    }
  });
  return {
    byId: resultById,
    aliases: {},
    rootLevelFields: resultRootLevel,
    maxNestedDepth: fullNormalized.maxNestedDepth
  };
};
exports.prepareFieldsForEisUpdate = prepareFieldsForEisUpdate;