"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.KNOWLEDGE_BASE_PATH = exports.KNOWLEDGE_BASE_FOLDER = exports.DOCS_PATH_PATTERN = void 0;
exports.cleanupKnowledgeBaseStep = cleanupKnowledgeBaseStep;
exports.indexKnowledgeBase = indexKnowledgeBase;
exports.stepSaveKnowledgeBase = stepSaveKnowledgeBase;
var _path = _interopRequireDefault(require("path"));
var _errors = require("../../../../../errors");
var _knowledge_base_index = require("../../knowledge_base_index");
var _types = require("../../../../../../common/types");
var _utils = require("../../utils");
var _epm = require("../../../../../../common/types/models/epm");
var _es_assets_reference = require("../../es_assets_reference");
var _license = require("../../../../license");
var _get_integration_knowledge_setting = require("../../get_integration_knowledge_setting");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const KNOWLEDGE_BASE_PATH = exports.KNOWLEDGE_BASE_PATH = 'docs/knowledge_base/';
const DOCS_PATH_PATTERN = exports.DOCS_PATH_PATTERN = '/docs/';
const KNOWLEDGE_BASE_FOLDER = exports.KNOWLEDGE_BASE_FOLDER = 'knowledge_base/';

/**
 * Extract knowledge base files directly from the package archive
 * This includes all .md files from the docs/ folder (including docs/knowledge_base/)
 */
async function extractKnowledgeBaseFromArchive(archiveIterator, pkgName, pkgVersion) {
  const knowledgeBaseItems = [];
  await archiveIterator.traverseEntries(async entry => {
    if (entry.buffer) {
      const content = entry.buffer.toString('utf8');

      // Determine the filename based on the path
      let fileName;
      const docsIndex = entry.path.indexOf(DOCS_PATH_PATTERN);
      if (docsIndex >= 0) {
        // Extract path relative to docs/ folder
        const pathAfterDocs = entry.path.substring(docsIndex + DOCS_PATH_PATTERN.length);

        // If it's in knowledge_base subfolder, remove that prefix too for backward compatibility
        if (pathAfterDocs.startsWith(KNOWLEDGE_BASE_FOLDER)) {
          fileName = pathAfterDocs.substring(KNOWLEDGE_BASE_FOLDER.length);
        } else {
          fileName = pathAfterDocs;
        }
      } else {
        // Fallback to basename
        fileName = _path.default.basename(entry.path);
      }
      knowledgeBaseItems.push({
        fileName,
        content
      });
    }
  }, entryPath => entryPath.includes(DOCS_PATH_PATTERN) && entryPath.endsWith('.md'));
  return knowledgeBaseItems;
}
async function stepSaveKnowledgeBase(context) {
  var _context$esReferences;
  const {
    packageInstallContext,
    esClient,
    savedObjectsClient,
    logger
  } = context;
  const {
    packageInfo,
    archiveIterator
  } = packageInstallContext;
  const esReferences = (_context$esReferences = context.esReferences) !== null && _context$esReferences !== void 0 ? _context$esReferences : [];
  logger.debug(`Knowledge base step: Starting for package ${packageInfo.name}@${packageInfo.version}`);
  const integrationKnowledgeEnabled = await (0, _get_integration_knowledge_setting.getIntegrationKnowledgeSetting)(savedObjectsClient);

  // Check if knowledge base installation is enabled via user setting
  if (!integrationKnowledgeEnabled) {
    logger.debug(`Knowledge base step: Skipping knowledge base save - integration knowledge enabled setting is disabled`);
    return {
      esReferences
    };
  }

  // Check if user has appropriate license for knowledge base functionality
  // You can adjust the license requirement as needed (e.g., isGoldPlus(), isPlatinum(), isEnterprise())
  if (!_license.licenseService.isEnterprise()) {
    logger.debug(`Knowledge base step: Skipping knowledge base save - requires Enterprise license`);
    return {
      esReferences
    };
  }
  return await indexKnowledgeBase(esReferences, savedObjectsClient, esClient, logger, packageInfo, archiveIterator);
}
async function indexKnowledgeBase(esReferences, savedObjectsClient, esClient, logger, packageInfo, archiveIterator, abortController) {
  // Extract knowledge base content directly from the archive
  const knowledgeBaseItems = await extractKnowledgeBaseFromArchive(archiveIterator, packageInfo.name, packageInfo.version);
  logger.debug(`Knowledge base step: Found ${knowledgeBaseItems.length} items to process for package ${packageInfo.name}@${packageInfo.version}`);

  // Save knowledge base content if present
  if (knowledgeBaseItems && knowledgeBaseItems.length > 0) {
    try {
      // Save knowledge base content - this handles both fresh installs and upgrades
      // by always deleting existing content for the package before saving new content
      const documentIds = await (0, _knowledge_base_index.saveKnowledgeBaseContentToIndex)({
        esClient,
        pkgName: packageInfo.name,
        pkgVersion: packageInfo.version,
        knowledgeBaseContent: knowledgeBaseItems,
        abortController
      });
      logger.debug(`Knowledge base step: Saved ${documentIds.length} documents to index for package ${packageInfo.name}@${packageInfo.version}`);

      // Add knowledge base asset references using the ES-generated document IDs
      const knowledgeBaseAssetRefs = documentIds.map(docId => ({
        id: docId,
        type: _epm.ElasticsearchAssetType.knowledgeBase
      }));

      // Update ES asset references to include knowledge base assets
      esReferences = await (0, _es_assets_reference.updateEsAssetReferences)(savedObjectsClient, packageInfo.name, esReferences, {
        assetsToAdd: knowledgeBaseAssetRefs
      });
    } catch (error) {
      throw new _errors.FleetError(`Error saving knowledge base content: ${error}`);
    }
  }
  return {
    esReferences
  };
}
async function cleanupKnowledgeBaseStep(context) {
  var _installedPkg$attribu;
  const {
    logger,
    esClient,
    installedPkg,
    retryFromLastState,
    force,
    initialState
  } = context;

  // Clean up knowledge base content during retry or rollback
  if (!force && retryFromLastState && initialState === _types.INSTALL_STATES.SAVE_KNOWLEDGE_BASE && esClient && installedPkg !== null && installedPkg !== void 0 && (_installedPkg$attribu = installedPkg.attributes) !== null && _installedPkg$attribu !== void 0 && _installedPkg$attribu.name) {
    logger.debug('Retry transition - clean up package knowledge base content');
    await (0, _utils.withPackageSpan)('Retry transition - clean up package knowledge base content', async () => {
      await (0, _knowledge_base_index.deletePackageKnowledgeBase)(esClient, installedPkg.attributes.name);
    });
  }
}