"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.addErrorToLatestFailedAttempts = addErrorToLatestFailedAttempts;
exports.clearLatestFailedAttempts = clearLatestFailedAttempts;
exports.createOrUpdateFailedInstallStatus = void 0;
var _semver = require("semver");
var _server = require("@kbn/core/server");
var _audit_logging = require("../../audit_logging");
var _constants = require("../../../constants");
var _get = require("./get");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const MAX_ATTEMPTS_TO_KEEP = 5;
function clearLatestFailedAttempts(installedVersion, latestAttempts = []) {
  return latestAttempts.filter(attempt => (0, _semver.lt)(installedVersion, attempt.target_version));
}
function addErrorToLatestFailedAttempts({
  error,
  createdAt,
  targetVersion,
  latestAttempts = []
}) {
  return [{
    created_at: createdAt,
    target_version: targetVersion,
    error: {
      name: error.name,
      message: error.message,
      stack: error.stack
    }
  }, ...latestAttempts].slice(0, MAX_ATTEMPTS_TO_KEEP);
}
const createOrUpdateFailedInstallStatus = async ({
  logger,
  savedObjectsClient,
  pkgName,
  pkgVersion,
  error,
  installSource = 'registry'
}) => {
  const installation = await (0, _get.getInstallationObject)({
    pkgName,
    savedObjectsClient
  });
  if (installation) {
    var _installation$attribu;
    _audit_logging.auditLoggingService.writeCustomSoAuditLog({
      action: 'update',
      id: pkgName,
      name: pkgName,
      savedObjectType: _constants.PACKAGES_SAVED_OBJECT_TYPE
    });
    const latestInstallFailedAttempts = addErrorToLatestFailedAttempts({
      error,
      targetVersion: pkgVersion,
      createdAt: new Date().toISOString(),
      latestAttempts: installation === null || installation === void 0 ? void 0 : (_installation$attribu = installation.attributes) === null || _installation$attribu === void 0 ? void 0 : _installation$attribu.latest_install_failed_attempts
    });
    try {
      return await savedObjectsClient.update(_constants.PACKAGES_SAVED_OBJECT_TYPE, pkgName, {
        latest_install_failed_attempts: latestInstallFailedAttempts
      });
    } catch (err) {
      logger.warn(`Error occurred while updating installation failed attempts: ${err}`);
    }
  } else {
    _audit_logging.auditLoggingService.writeCustomSoAuditLog({
      action: 'create',
      id: pkgName,
      name: pkgName,
      savedObjectType: _constants.PACKAGES_SAVED_OBJECT_TYPE
    });
    const installFailedAttempts = addErrorToLatestFailedAttempts({
      error,
      targetVersion: pkgVersion,
      createdAt: new Date().toISOString(),
      latestAttempts: []
    });
    const savedObject = {
      installed_kibana: [],
      installed_es: [],
      package_assets: [],
      name: pkgName,
      version: pkgVersion,
      install_version: pkgVersion,
      install_status: 'install_failed',
      install_started_at: new Date().toISOString(),
      verification_status: 'unknown',
      latest_install_failed_attempts: installFailedAttempts,
      es_index_patterns: {},
      install_source: installSource
    };
    try {
      return await savedObjectsClient.create(_constants.PACKAGES_SAVED_OBJECT_TYPE, savedObject, {
        id: pkgName,
        overwrite: true
      });
    } catch (err) {
      if (!_server.SavedObjectsErrorHelpers.isNotFoundError(err)) {
        logger.error(`Failed to create package installation: ${err}`);
      }
    }
  }
};
exports.createOrUpdateFailedInstallStatus = createOrUpdateFailedInstallStatus;