"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getDatasetName = exports.findDataStreamsFromDifferentPackages = exports.checkExistingDataStreamsAreFromDifferentPackage = void 0;
exports.installAssetsForInputPackagePolicy = installAssetsForInputPackagePolicy;
exports.isInputPackageDatasetUsedByMultiplePolicies = void 0;
exports.removeAssetsForInputPackagePolicy = removeAssetsForInputPackagePolicy;
var _constants = require("../../../../common/constants");
var _errors = require("../../../errors");
var _ = require("../..");
var Registry = _interopRequireWildcard(require("../registry"));
var _archive_iterator = require("../archive/archive_iterator");
var _services = require("../../../../common/services");
var _template = require("../elasticsearch/template/template");
var _get = require("./get");
var _install_index_template_pipeline = require("./install_index_template_pipeline");
var _es_assets_reference = require("./es_assets_reference");
var _remove = require("./remove");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getDatasetName = packagePolicyInput => {
  var _packagePolicyInput$, _packagePolicyInput$$, _packagePolicyInput$$2;
  return (_packagePolicyInput$ = packagePolicyInput[0].streams[0]) === null || _packagePolicyInput$ === void 0 ? void 0 : (_packagePolicyInput$$ = _packagePolicyInput$.vars) === null || _packagePolicyInput$$ === void 0 ? void 0 : (_packagePolicyInput$$2 = _packagePolicyInput$$[_constants.DATASET_VAR_NAME]) === null || _packagePolicyInput$$2 === void 0 ? void 0 : _packagePolicyInput$$2.value;
};
exports.getDatasetName = getDatasetName;
const findDataStreamsFromDifferentPackages = async (datasetName, pkgInfo, esClient, dataStreamType) => {
  const [dataStream] = (0, _services.getNormalizedDataStreams)(pkgInfo, datasetName, dataStreamType);
  const existingDataStreams = await _.dataStreamService.getMatchingDataStreams(esClient, {
    type: dataStream.type,
    dataset: datasetName
  });
  return {
    dataStream,
    existingDataStreams
  };
};
exports.findDataStreamsFromDifferentPackages = findDataStreamsFromDifferentPackages;
const checkExistingDataStreamsAreFromDifferentPackage = (pkgInfo, existingDataStreams) => {
  return (existingDataStreams || []).some(ds => {
    var _ds$_meta, _ds$_meta$package;
    return ((_ds$_meta = ds._meta) === null || _ds$_meta === void 0 ? void 0 : (_ds$_meta$package = _ds$_meta.package) === null || _ds$_meta$package === void 0 ? void 0 : _ds$_meta$package.name) !== pkgInfo.name;
  });
};
exports.checkExistingDataStreamsAreFromDifferentPackage = checkExistingDataStreamsAreFromDifferentPackage;
const isInputPackageDatasetUsedByMultiplePolicies = (packagePolicies, datasetName, pkgName) => {
  const allStreams = packagePolicies.filter(packagePolicy => {
    var _packagePolicy$packag, _packagePolicy$packag2;
    return (packagePolicy === null || packagePolicy === void 0 ? void 0 : (_packagePolicy$packag = packagePolicy.package) === null || _packagePolicy$packag === void 0 ? void 0 : _packagePolicy$packag.name) === pkgName || (packagePolicy === null || packagePolicy === void 0 ? void 0 : (_packagePolicy$packag2 = packagePolicy.package) === null || _packagePolicy$packag2 === void 0 ? void 0 : _packagePolicy$packag2.type) === 'input';
  }).flatMap(packagePolicy => {
    var _packagePolicy$inputs, _packagePolicy$inputs2;
    return (_packagePolicy$inputs = packagePolicy === null || packagePolicy === void 0 ? void 0 : (_packagePolicy$inputs2 = packagePolicy.inputs[0]) === null || _packagePolicy$inputs2 === void 0 ? void 0 : _packagePolicy$inputs2.streams) !== null && _packagePolicy$inputs !== void 0 ? _packagePolicy$inputs : [];
  });
  const filtered = allStreams.filter(stream => {
    var _stream$vars, _stream$vars$DATASET_;
    return ((_stream$vars = stream.vars) === null || _stream$vars === void 0 ? void 0 : (_stream$vars$DATASET_ = _stream$vars[_constants.DATASET_VAR_NAME]) === null || _stream$vars$DATASET_ === void 0 ? void 0 : _stream$vars$DATASET_.value) === datasetName;
  });
  return filtered.length > 1;
};

// install the assets needed for inputs type packages
exports.isInputPackageDatasetUsedByMultiplePolicies = isInputPackageDatasetUsedByMultiplePolicies;
async function installAssetsForInputPackagePolicy(opts) {
  var _packagePolicy$inputs3, _packagePolicy$inputs4, _packagePolicy$inputs5;
  const {
    pkgInfo,
    logger,
    packagePolicy,
    esClient,
    soClient,
    force
  } = opts;
  if (pkgInfo.type !== 'input') return;
  const datasetName = getDatasetName(packagePolicy.inputs);
  const dataStreamType = ((_packagePolicy$inputs3 = packagePolicy.inputs[0].streams[0].vars) === null || _packagePolicy$inputs3 === void 0 ? void 0 : (_packagePolicy$inputs4 = _packagePolicy$inputs3[_constants.DATA_STREAM_TYPE_VAR_NAME]) === null || _packagePolicy$inputs4 === void 0 ? void 0 : _packagePolicy$inputs4.value) || ((_packagePolicy$inputs5 = packagePolicy.inputs[0].streams[0].data_stream) === null || _packagePolicy$inputs5 === void 0 ? void 0 : _packagePolicy$inputs5.type) || 'logs';
  const {
    dataStream,
    existingDataStreams
  } = await findDataStreamsFromDifferentPackages(datasetName, pkgInfo, esClient, dataStreamType);
  if (existingDataStreams.length) {
    const existingDataStreamsAreFromDifferentPackage = checkExistingDataStreamsAreFromDifferentPackage(pkgInfo, existingDataStreams);
    if (existingDataStreamsAreFromDifferentPackage && !force) {
      // user has opted to send data to an existing data stream which is managed by another
      // package. This means certain custom setting such as elasticsearch settings
      // defined by the package will not have been applied which could lead
      // to unforeseen circumstances, so force flag must be used.
      const streamIndexPattern = _.dataStreamService.streamPartsToIndexPattern({
        type: dataStream.type,
        dataset: datasetName
      });
      throw new _errors.PackagePolicyValidationError(`Datastreams matching "${streamIndexPattern}" already exist and are not managed by this package, force flag is required`);
    } else {
      logger.info(`Data stream for dataset ${datasetName} already exists, skipping index template creation`);
      return;
    }
  }
  const existingIndexTemplate = await _.dataStreamService.getMatchingIndexTemplate(esClient, {
    type: dataStream.type,
    dataset: datasetName
  });
  if (existingIndexTemplate) {
    var _existingIndexTemplat, _existingIndexTemplat2;
    const indexTemplateOwnedByDifferentPackage = ((_existingIndexTemplat = existingIndexTemplate._meta) === null || _existingIndexTemplat === void 0 ? void 0 : (_existingIndexTemplat2 = _existingIndexTemplat.package) === null || _existingIndexTemplat2 === void 0 ? void 0 : _existingIndexTemplat2.name) !== pkgInfo.name;
    if (indexTemplateOwnedByDifferentPackage && !force) {
      // index template already exists but there is no data stream yet
      // we do not want to override the index template
      throw new _errors.PackagePolicyValidationError(`Index template "${dataStream.type}-${datasetName}" already exist and is not managed by this package, force flag is required`);
    } else {
      logger.info(`Index template "${dataStream.type}-${datasetName}" already exists, skipping index template creation`);
      return;
    }
  }
  const installedPkgWithAssets = await (0, _get.getInstalledPackageWithAssets)({
    savedObjectsClient: soClient,
    pkgName: pkgInfo.name,
    logger
  });
  let packageInstallContext;
  if (!installedPkgWithAssets) {
    throw new _errors.PackageNotFoundError(`Error while creating index templates: unable to find installed package ${pkgInfo.name}`);
  }
  try {
    if (installedPkgWithAssets.installation.version !== pkgInfo.version) {
      const pkg = await Registry.getPackage(pkgInfo.name, pkgInfo.version, {
        ignoreUnverified: force
      });
      const archiveIterator = (0, _archive_iterator.createArchiveIteratorFromMap)(pkg.assetsMap);
      packageInstallContext = {
        packageInfo: pkg.packageInfo,
        paths: pkg.paths,
        archiveIterator
      };
    } else {
      const archiveIterator = (0, _archive_iterator.createArchiveIteratorFromMap)(installedPkgWithAssets.assetsMap);
      packageInstallContext = {
        packageInfo: installedPkgWithAssets.packageInfo,
        paths: installedPkgWithAssets.paths,
        archiveIterator
      };
    }
    await (0, _install_index_template_pipeline.installIndexTemplatesAndPipelines)({
      installedPkg: installedPkgWithAssets.installation,
      packageInstallContext,
      esReferences: installedPkgWithAssets.installation.installed_es || [],
      savedObjectsClient: soClient,
      esClient,
      logger,
      onlyForDataStreams: [dataStream]
    });
    // Upate ES index patterns
    await (0, _es_assets_reference.optimisticallyAddEsAssetReferences)(soClient, installedPkgWithAssets.installation.name, [], (0, _template.generateESIndexPatterns)([dataStream]));
  } catch (error) {
    logger.warn(`installAssetsForInputPackagePolicy error: ${error}`);
  }
}

// Remove the assets installed for input-type packages
async function removeAssetsForInputPackagePolicy(opts) {
  const {
    logger,
    packageInfo,
    esClient,
    savedObjectsClient,
    datasetName
  } = opts;
  if (packageInfo.type === 'input' && packageInfo.status === 'installed') {
    logger.info(`Removing assets for input package ${packageInfo.name}:${packageInfo.version}`);
    try {
      const installation = await (0, _get.getInstallation)({
        savedObjectsClient,
        pkgName: packageInfo.name
      });
      if (!installation) {
        throw new _errors.FleetError(`${packageInfo.name} is not installed`);
      }
      const {
        installed_es: installedEs,
        installed_kibana: installedKibana,
        es_index_patterns: esIndexPatterns
      } = installation;

      // regex matching names with word boundary, allows to match `generic` and not `generic1`
      const regex = new RegExp(`${datasetName}\\b`);
      const filteredInstalledEs = installedEs.filter(asset => asset.id.search(regex) > -1);
      const filteredInstalledKibana = installedKibana.filter(asset => asset.id.search(regex) > -1);
      const filteredEsIndexPatterns = {};
      if (esIndexPatterns) {
        filteredEsIndexPatterns[datasetName] = esIndexPatterns[datasetName];
      }
      const installationToDelete = {
        ...installation,
        installed_es: filteredInstalledEs,
        installed_kibana: filteredInstalledKibana,
        es_index_patterns: filteredEsIndexPatterns,
        package_assets: []
      };
      await (0, _remove.cleanupAssets)(datasetName, installationToDelete, installation, esClient, savedObjectsClient);
    } catch (error) {
      logger.error(`Failed to remove assets for input package ${packageInfo.name}:${packageInfo.version}: ${error.message}`);
    }
  }
}