"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.NO_ROLLBACK_ERROR_MESSAGE = exports.EXPIRED_ROLLBACK_ERROR_MESSAGE = void 0;
exports.getValidRollbacks = getValidRollbacks;
exports.sendRollbackAgentAction = sendRollbackAgentAction;
exports.sendRollbackAgentsActions = sendRollbackAgentsActions;
var _errors = require("../../errors");
var _constants = require("../../constants");
var _agent_namespaces = require("../spaces/agent_namespaces");
var _get_current_namespace = require("../spaces/get_current_namespace");
var _license = require("../license");
var _constants2 = require("../../../common/constants");
var _services = require("../../../common/services");
var _crud = require("./crud");
var _rollback_action_runner = require("./rollback_action_runner");
var _actions = require("./actions");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const NO_ROLLBACK_ERROR_MESSAGE = exports.NO_ROLLBACK_ERROR_MESSAGE = 'upgrade rollback not available for agent';
const EXPIRED_ROLLBACK_ERROR_MESSAGE = exports.EXPIRED_ROLLBACK_ERROR_MESSAGE = 'upgrade rollback window has expired';
function checkLicense() {
  if (!_license.licenseService.hasAtLeast(_constants2.LICENSE_FOR_AGENT_ROLLBACK)) {
    throw new _errors.FleetUnauthorizedError(`Agent upgrade rollback requires an ${_constants2.LICENSE_FOR_AGENT_ROLLBACK} license.`);
  }
}
function getValidRollbacks(agent) {
  var _agent$upgrade;
  const now = new Date();
  return (((_agent$upgrade = agent.upgrade) === null || _agent$upgrade === void 0 ? void 0 : _agent$upgrade.rollbacks) || []).filter(rollback => {
    const validUntil = new Date(rollback.valid_until);
    return validUntil > now;
  });
}
async function sendRollbackAgentAction(soClient, esClient, agent) {
  var _agent$upgrade2;
  checkLicense();
  if (agent.unenrollment_started_at || agent.unenrolled_at) {
    throw new _errors.AgentRollbackError('cannot roll back an unenrolling or unenrolled agent');
  }
  if ((0, _services.isAgentUpgrading)(agent)) {
    throw new _errors.AgentRollbackError('cannot roll back an upgrading agent');
  }
  const rollbacks = ((_agent$upgrade2 = agent.upgrade) === null || _agent$upgrade2 === void 0 ? void 0 : _agent$upgrade2.rollbacks) || [];
  if (rollbacks.length === 0) {
    throw new _errors.AgentRollbackError(NO_ROLLBACK_ERROR_MESSAGE);
  }
  const validRollbacks = getValidRollbacks(agent);
  if (validRollbacks.length === 0) {
    throw new _errors.AgentRollbackError(EXPIRED_ROLLBACK_ERROR_MESSAGE);
  }
  const agentPolicy = await (0, _crud.getAgentPolicyForAgent)(soClient, esClient, agent.id);
  if (agentPolicy !== null && agentPolicy !== void 0 && agentPolicy.is_managed) {
    throw new _errors.HostedAgentPolicyRestrictionRelatedError(`Cannot roll back upgrade for agent ${agent.id} on hosted agent policy ${agentPolicy.id}`);
  }
  const now = new Date().toISOString();
  const currentSpaceId = (0, _get_current_namespace.getCurrentNamespace)(soClient);
  const data = {
    version: validRollbacks[0].version,
    rollback: true
  };
  const action = await (0, _actions.createAgentAction)(esClient, soClient, {
    agents: [agent.id],
    created_at: now,
    data,
    ack_data: data,
    type: 'UPGRADE',
    namespaces: [currentSpaceId]
  });
  await (0, _crud.updateAgent)(esClient, agent.id, {
    upgraded_at: null,
    upgrade_started_at: now
  });
  return action.id;
}
async function sendRollbackAgentsActions(soClient, esClient, options) {
  checkLicense();
  const currentSpaceId = (0, _get_current_namespace.getCurrentNamespace)(soClient);
  const errors = {};
  let givenAgents = [];
  if ('agents' in options) {
    givenAgents = options.agents;
  } else if ('agentIds' in options) {
    const maybeAgents = await (0, _crud.getAgentsById)(esClient, soClient, options.agentIds);
    for (const maybeAgent of maybeAgents) {
      if ('notFound' in maybeAgent) {
        errors[maybeAgent.id] = new _errors.AgentNotFoundError(`Agent ${maybeAgent.id} not found`);
      } else {
        givenAgents.push(maybeAgent);
      }
    }
  } else if ('kuery' in options) {
    var _options$batchSize, _options$includeInact;
    const batchSize = (_options$batchSize = options.batchSize) !== null && _options$batchSize !== void 0 ? _options$batchSize : _constants.SO_SEARCH_LIMIT;
    const namespaceFilter = await (0, _agent_namespaces.agentsKueryNamespaceFilter)(currentSpaceId);
    const kuery = namespaceFilter ? `${namespaceFilter} AND ${options.kuery}` : options.kuery;
    const res = await (0, _crud.getAgentsByKuery)(esClient, soClient, {
      kuery,
      showAgentless: options.showAgentless,
      showInactive: (_options$includeInact = options.includeInactive) !== null && _options$includeInact !== void 0 ? _options$includeInact : false,
      page: 1,
      perPage: batchSize
    });
    if (res.total <= batchSize) {
      givenAgents = res.agents;
    } else {
      // Upgrade rollback returns all action IDs (one per rollback version group)
      // Process all batches synchronously to collect all action IDs
      const runner = new _rollback_action_runner.RollbackActionRunner(esClient, soClient, {
        ...options,
        batchSize,
        total: res.total,
        spaceId: currentSpaceId
      }, {
        pitId: await (0, _crud.openPointInTime)(esClient)
      });
      // Process all batches synchronously to collect all action IDs
      await runner.processAgentsInBatches();
      // Return all collected action IDs
      return {
        actionIds: runner.getAllActionIds()
      };
    }
  }
  const result = await (0, _rollback_action_runner.rollbackBatch)(esClient, givenAgents, errors, {}, [currentSpaceId]);
  return {
    actionIds: result.actionIds
  };
}