"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getProjectRoutingFromEsqlQuery = getProjectRoutingFromEsqlQuery;
var _esqlAst = require("@kbn/esql-ast");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Extracts the project routing value from an ES|QL query string.
 *
 * @param queryString - The ES|QL query string to parse
 * @returns The project routing value if found, undefined otherwise
 *
 * @example
 * getProjectRoutingFromEsqlQuery('SET project_routing = "_alias: *"; FROM my_index')
 * // Returns: '_alias: *'
 *
 * getProjectRoutingFromEsqlQuery('FROM my_index')
 * // Returns: undefined
 */
function getProjectRoutingFromEsqlQuery(queryString) {
  try {
    var _parsedQuery$ast$head;
    const parsedQuery = _esqlAst.EsqlQuery.fromSrc(queryString);
    const headerInstructions = (_parsedQuery$ast$head = parsedQuery.ast.header) !== null && _parsedQuery$ast$head !== void 0 ? _parsedQuery$ast$head : [];
    for (const instruction of headerInstructions) {
      const instructionFunction = instruction.args.find(_esqlAst.isFunctionExpression);
      if (!instructionFunction) continue;
      const identifierArg = instructionFunction.args.find(_esqlAst.isIdentifier);
      if ((identifierArg === null || identifierArg === void 0 ? void 0 : identifierArg.name) !== 'project_routing') continue;
      const valueArg = instructionFunction.args[1];
      if (!valueArg || Array.isArray(valueArg)) continue;
      if ((0, _esqlAst.isLiteral)(valueArg)) {
        // For string literals, extract the unquoted value
        if (valueArg.literalType === 'keyword') {
          return valueArg.valueUnquoted;
        }
        // For other literal types, use the printer to get proper string representation
        return _esqlAst.LeafPrinter.literal(valueArg);
      }
    }
  } catch (error) {
    // Silently handle parsing errors
  }
  return undefined;
}