"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getIndexPatternFromESQLQuery = getIndexPatternFromESQLQuery;
var _esqlAst = require("@kbn/esql-ast");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

function getSourcesFromAst(commands) {
  const sourceCommand = commands.find(({
    name
  }) => ['from', 'ts'].includes(name));
  if (!sourceCommand) {
    return [];
  }
  const args = sourceCommand.args;
  return args.filter(arg => arg.sourceType === 'index').map(index => index.name);
}
function extractSubquerySources(sourceCommand) {
  const subqueryArgs = sourceCommand.args.filter(_esqlAst.isSubQuery);
  const subquerySources = [];
  for (const subquery of subqueryArgs) {
    const sources = getSourcesFromAst(subquery.child.commands);
    subquerySources.push(...sources);
  }
  return subquerySources;
}

/**
 * Retrieves the index pattern from an ES|QL query using AST parsing.
 * Handles both main queries and subqueries within FROM/TS commands.
 *
 * @param esql - The ES|QL query string to parse
 * @returns Comma-separated string of unique index names, or empty string if no sources found
 */
function getIndexPatternFromESQLQuery(esql) {
  if (!(esql !== null && esql !== void 0 && esql.trim())) {
    return '';
  }
  const {
    root
  } = _esqlAst.Parser.parse(esql);
  const allSources = [];

  // Get sources from main query
  const mainSources = getSourcesFromAst(root.commands);
  allSources.push(...mainSources);

  // Get sources from subqueries
  const sourceCommand = root.commands.find(({
    name
  }) => ['from', 'ts'].includes(name));
  if (sourceCommand) {
    const subquerySources = extractSubquerySources(sourceCommand);
    allSources.push(...subquerySources);
  }

  // Remove duplicates
  const uniqueSources = [...new Set(allSources)];
  return uniqueSources.join(',');
}