"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getESQLStatsQueryMeta = exports.constructCascadeQuery = exports.appendFilteringWhereClauseForCascadeLayout = void 0;
exports.getFieldParamDefinition = getFieldParamDefinition;
exports.getStatsCommandToOperateOn = getStatsCommandToOperateOn;
exports.getStatsGroupFieldType = getStatsGroupFieldType;
exports.isSupportedFieldType = void 0;
exports.mutateQueryStatsGrouping = mutateQueryStatsGrouping;
var _esqlAst = require("@kbn/esql-ast");
var _extract_categorize_tokens = require("./extract_categorize_tokens");
var _utils = require("./append_to_query/utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

// list of stats functions we support for grouping in the cascade experience
const SUPPORTED_STATS_COMMAND_OPTION_FUNCTIONS = ['categorize'];
const isSupportedStatsFunction = fnName => SUPPORTED_STATS_COMMAND_OPTION_FUNCTIONS.includes(fnName);
// helper for removing backticks from field names of function names
const removeBackticks = str => str.replace(/`/g, '');

/**
 * constrains the field value type to be one of the supported field value types, else we process as a string literal when building the expression
 */
const isSupportedFieldType = fieldType => {
  return _utils.PARAM_TYPES_NO_NEED_IMPLICIT_STRING_CASTING.includes(fieldType) && Object.keys(_esqlAst.Builder.expression.literal).includes(fieldType);
};

// if value is a text or keyword field and it's not "aggregatable", we opt to use match phrase for the where command
exports.isSupportedFieldType = isSupportedFieldType;
const requiresMatchPhrase = (fieldName, dataViewFields) => {
  var _dataViewField$esType, _dataViewField$esType2;
  const dataViewField = dataViewFields.getByName(fieldName);
  return ((dataViewField === null || dataViewField === void 0 ? void 0 : (_dataViewField$esType = dataViewField.esTypes) === null || _dataViewField$esType === void 0 ? void 0 : _dataViewField$esType.includes('text')) || (dataViewField === null || dataViewField === void 0 ? void 0 : (_dataViewField$esType2 = dataViewField.esTypes) === null || _dataViewField$esType2 === void 0 ? void 0 : _dataViewField$esType2.includes('keyword'))) && !(dataViewField !== null && dataViewField !== void 0 && dataViewField.aggregatable);
};
function getStatsCommandToOperateOn(esqlQuery) {
  if (esqlQuery.errors.length) {
    return null;
  }
  const statsCommands = Array.from(_esqlAst.mutate.commands.stats.list(esqlQuery.ast));
  if (statsCommands.length === 0) {
    return null;
  }

  // accounting for the possibility of multiple stats commands in the query,
  // we always want to operate on the last stats command
  const summarizedStatsCommand = _esqlAst.mutate.commands.stats.summarizeCommand(esqlQuery, statsCommands[statsCommands.length - 1]);
  return summarizedStatsCommand;
}
function getESQLQueryDataSourceCommand(esqlQuery) {
  return _esqlAst.mutate.generic.commands.find(esqlQuery.ast, cmd => cmd.name === 'from' || cmd.name === 'ts');
}

/**
 * Returns runtime fields that are created within the query by the STATS command in the query
 */
function getStatsCommandRuntimeFields(esqlQuery) {
  return Array.from(_esqlAst.mutate.commands.stats.summarize(esqlQuery)).map(command => command.newFields);
}

/**
 * Returns the summary of the stats command at the given command index in the esql query
 */
function getStatsCommandAtIndexSummary(esqlQuery, commandIndex) {
  const declarationCommand = _esqlAst.mutate.commands.stats.byIndex(esqlQuery.ast, commandIndex);
  if (!declarationCommand) {
    return null;
  }
  return _esqlAst.mutate.commands.stats.summarizeCommand(esqlQuery, declarationCommand);
}
function getFieldParamDefinition(fieldName, fieldTerminals, esqlVariables) {
  const fieldParamDef = fieldTerminals.find(arg => arg.text === fieldName && arg.type === 'literal' && arg.literalType === 'param');
  if (fieldParamDef) {
    const controlVariable = esqlVariables === null || esqlVariables === void 0 ? void 0 : esqlVariables.find(variable => variable.key === fieldParamDef.value);
    if (!controlVariable) {
      throw new Error(`The control variable for the "${fieldName}" column was not found`);
    }
    return controlVariable.value;
  }
}
function getStatsGroupFieldType(groupByFields) {
  if (!groupByFields) {
    return undefined;
  }
  return groupByFields.definition.type === 'function' ? groupByFields.definition.name : groupByFields.definition.type;
}
const getESQLStatsQueryMeta = queryString => {
  var _summarizedStatsComma;
  const groupByFields = [];
  const appliedFunctions = [];
  const esqlQuery = _esqlAst.EsqlQuery.fromSrc(queryString);
  const dataSourceCommand = getESQLQueryDataSourceCommand(esqlQuery);
  const summarizedStatsCommand = getStatsCommandToOperateOn(esqlQuery);
  if (dataSourceCommand !== null && dataSourceCommand !== void 0 && dataSourceCommand.args.some(_esqlAst.isSubQuery) || !summarizedStatsCommand || Object.keys((_summarizedStatsComma = summarizedStatsCommand === null || summarizedStatsCommand === void 0 ? void 0 : summarizedStatsCommand.grouping) !== null && _summarizedStatsComma !== void 0 ? _summarizedStatsComma : {}).length === 0) {
    return {
      groupByFields,
      appliedFunctions
    };
  }

  // get all the new fields created by the stats commands in the query,
  // so we might tell if the command we are operating on is referencing a field that was defined by a preceding command
  const statsCommandRuntimeFields = getStatsCommandRuntimeFields(esqlQuery);
  const grouping = Object.values(summarizedStatsCommand.grouping);
  for (let j = 0; j < grouping.length; j++) {
    const group = grouping[j];
    if (!group.definition) {
      // query received is malformed without complete grouping definition, there's no need to proceed further
      return {
        groupByFields: [],
        appliedFunctions: []
      };
    }
    const groupFieldName = removeBackticks(group.field);
    let groupFieldNode = group;
    const groupDeclarationCommandIndex = statsCommandRuntimeFields.findIndex(field => field.has(groupFieldName));
    let groupDeclarationCommandSummary = null;
    if (groupDeclarationCommandIndex >= 0 && (groupDeclarationCommandSummary = getStatsCommandAtIndexSummary(esqlQuery, groupDeclarationCommandIndex))) {
      // update the group field node to it's actual definition
      groupFieldNode = groupDeclarationCommandSummary.grouping[groupFieldName];
    }

    // check if there is a where command after the operating stats command targeting any of it's grouping options
    const whereCommandGroupFieldSearch = esqlQuery.ast.commands.slice(groupDeclarationCommandIndex).find(cmd => {
      if (cmd.name !== 'where') {
        return false;
      }
      let found = false;
      _esqlAst.Walker.walk(cmd, {
        visitIdentifier: node => {
          if (found) {
            return;
          }
          if (node.name === groupFieldName) {
            found = true;
          }
        }
      });
      return found;
    });
    if (whereCommandGroupFieldSearch) {
      if (groupByFields.length > 0) {
        // if there's a where command targeting the group in this current iteration,
        // then this specific query can only be grouped by the current group, pivoting on any other columns though they exist
        // in the query would be invalid, hence we clear out any previously added group by fields since they are no longer valid
        groupByFields.splice(0, groupByFields.length);
      }

      // add the current group and break out of the loop
      // since there's no need to continue processing other groups
      // as they are not valid in this context
      groupByFields.push({
        field: groupFieldName,
        type: getStatsGroupFieldType(groupFieldNode)
      });
      break;
    }
    if ((0, _esqlAst.isFunctionExpression)(groupFieldNode.definition)) {
      const functionName = groupFieldNode.definition.name;
      if (!isSupportedStatsFunction(functionName)) {
        continue;
      }
    }
    groupByFields.push({
      field: groupFieldName,
      type: getStatsGroupFieldType(groupFieldNode)
    });
  }
  Object.values(summarizedStatsCommand.aggregates).forEach(aggregate => {
    var _operator$name, _operator;
    appliedFunctions.push({
      identifier: removeBackticks(aggregate.field),
      // we remove backticks to have a clean identifier that gets displayed in the UI
      aggregation: (_operator$name = (_operator = aggregate.definition.operator) === null || _operator === void 0 ? void 0 : _operator.name) !== null && _operator$name !== void 0 ? _operator$name : aggregate.definition.text
    });
  });
  return {
    groupByFields,
    appliedFunctions
  };
};
exports.getESQLStatsQueryMeta = getESQLStatsQueryMeta;
/**
 * Constructs a cascade query from the provided query, based on the node type, node path and node path map.
 */
const constructCascadeQuery = ({
  query,
  dataView,
  esqlVariables,
  nodeType,
  nodePath,
  nodePathMap
}) => {
  var _summarizedStatsComma2;
  const EditorESQLQuery = _esqlAst.EsqlQuery.fromSrc(query.esql);
  if (EditorESQLQuery.errors.length) {
    throw new Error('Query is malformed');
  }
  const summarizedStatsCommand = getStatsCommandToOperateOn(EditorESQLQuery);
  if (!summarizedStatsCommand || Object.keys((_summarizedStatsComma2 = summarizedStatsCommand === null || summarizedStatsCommand === void 0 ? void 0 : summarizedStatsCommand.grouping) !== null && _summarizedStatsComma2 !== void 0 ? _summarizedStatsComma2 : {}).length === 0) {
    throw new Error('Query does not have a valid stats command with grouping options');
  }
  if (nodeType === 'leaf') {
    var _fieldDeclarationComm;
    const pathSegment = nodePath[nodePath.length - 1];

    // we make an initial assumption that the field was declared by the stats command being operated on
    let fieldDeclarationCommandSummary = summarizedStatsCommand;

    // if field name is not marked as a new field then we want ascertain it wasn't created by a preceding stats command
    if (!fieldDeclarationCommandSummary.newFields.has(pathSegment)) {
      var _groupDeclarationComm;
      const statsCommandRuntimeFields = getStatsCommandRuntimeFields(EditorESQLQuery);
      const groupDeclarationCommandIndex = statsCommandRuntimeFields.findIndex(field => field.has(pathSegment));
      const groupIsStatsDeclaredRuntimeField = groupDeclarationCommandIndex >= 0;
      let groupDeclarationCommandSummary = null;
      if (groupIsStatsDeclaredRuntimeField) {
        groupDeclarationCommandSummary = getStatsCommandAtIndexSummary(EditorESQLQuery, groupDeclarationCommandIndex);
      }
      fieldDeclarationCommandSummary = (_groupDeclarationComm = groupDeclarationCommandSummary) !== null && _groupDeclarationComm !== void 0 ? _groupDeclarationComm : fieldDeclarationCommandSummary;
    }
    const groupValue = (_fieldDeclarationComm = fieldDeclarationCommandSummary.grouping[pathSegment]) !== null && _fieldDeclarationComm !== void 0 ? _fieldDeclarationComm :
    // when a column name is not assigned on using a grouping function, one is created automatically from the function expression that includes backticks
    fieldDeclarationCommandSummary.grouping[`\`${pathSegment}\``];
    if (
    // check if we have a value for the path segment in the node path map to match on
    !(groupValue && nodePathMap[pathSegment] !== undefined)) {
      throw new Error(`The "${pathSegment}" field is not operable`);
    }
    const operatingStatsCommandIndex = EditorESQLQuery.ast.commands.findIndex(cmd => cmd.text === summarizedStatsCommand.command.text);
    if ((0, _esqlAst.isColumn)(groupValue.definition)) {
      return handleStatsByColumnLeafOperation(EditorESQLQuery, operatingStatsCommandIndex, groupValue, dataView.fields, esqlVariables, nodePathMap[pathSegment]);
    } else if ((0, _esqlAst.isFunctionExpression)(groupValue.definition)) {
      switch (groupValue.definition.name) {
        case 'categorize':
          {
            return handleStatsByCategorizeLeafOperation(EditorESQLQuery, operatingStatsCommandIndex, groupValue, esqlVariables, nodePathMap);
          }
        default:
          {
            throw new Error(`The "${groupValue.definition.name}" function is not supported for leaf node operations`);
          }
      }
    }
  } else if (nodeType === 'group') {
    throw new Error('Group node operations are not yet supported');
  }
};

/**
 * @description adds a where command with current value for a matched column option as a side-effect on the passed query,
 * helps us with fetching documents that match the value of a specified column on a stats operation in the data cascade experience.
 */
exports.constructCascadeQuery = constructCascadeQuery;
function handleStatsByColumnLeafOperation(editorQuery, operatingStatsCommandIndex, columnNode, dataViewFields, esqlVariables, operationValue) {
  // create a new query to populate with the cascade operation query
  const cascadeOperationQuery = _esqlAst.EsqlQuery.fromSrc('');

  // include all the existing commands up to the operating stats command in the cascade operation query
  editorQuery.ast.commands.slice(0, operatingStatsCommandIndex + 1).forEach((cmd, idx, arr) => {
    if (idx === arr.length - 1 && cmd.name === 'stats') {
      // We know the operating stats command is the last command in the array,
      // so we modify it into an INLINE STATS command
      _esqlAst.mutate.generic.commands.append(cascadeOperationQuery.ast, _esqlAst.synth.cmd(`INLINE ${_esqlAst.BasicPrettyPrinter.print(cmd)}`, {
        withFormatting: false
      }));
    } else {
      _esqlAst.mutate.generic.commands.append(cascadeOperationQuery.ast, cmd);
    }
  });
  let operationColumnName = columnNode.definition.name;
  let operationColumnNameParamValue;
  if (operationColumnNameParamValue = getFieldParamDefinition(operationColumnName, columnNode.terminals, esqlVariables)) {
    if (typeof operationColumnNameParamValue === 'string') {
      // we expect the operation column name parameter value to be a string, so we check for that and update the operation column name to the param value if it is
      operationColumnName = operationColumnNameParamValue;
    }
  }
  const shouldUseMatchPhrase = requiresMatchPhrase(operationColumnName, dataViewFields);

  // build a where command with match expressions for the selected column
  const filterCommand = _esqlAst.Builder.command({
    name: 'where',
    args: [shouldUseMatchPhrase ? _esqlAst.Builder.expression.func.call('match_phrase', [_esqlAst.Builder.identifier({
      name: operationColumnName
    }), _esqlAst.Builder.expression.literal.string(operationValue)]) : _esqlAst.Builder.expression.func.binary('==', [_esqlAst.Builder.expression.column({
      args: [_esqlAst.Builder.identifier({
        name: operationColumnName
      })]
    }), Number.isNaN(Number(operationValue)) ? _esqlAst.Builder.expression.literal.string(operationValue) : _esqlAst.Builder.expression.literal.integer(Number(operationValue))])]
  });
  _esqlAst.mutate.generic.commands.append(cascadeOperationQuery.ast, filterCommand);
  return {
    esql: _esqlAst.BasicPrettyPrinter.print(cascadeOperationQuery.ast)
  };
}

/**
 * Handles the stats command for a leaf operation that contains a categorize function by modifying the query and adding necessary commands.
 */
function handleStatsByCategorizeLeafOperation(editorQuery, operatingStatsCommandIndex, categorizeCommandNode, esqlVariables, nodePathMap) {
  // create a new query to populate with the cascade operation query
  const cascadeOperationQuery = _esqlAst.EsqlQuery.fromSrc('');

  // include all the existing commands right up until the operating stats command
  editorQuery.ast.commands.slice(0, operatingStatsCommandIndex).forEach((cmd, idx, arr) => {
    if (idx === arr.length - 1 && cmd.name === 'stats') {
      // however, when the last command is a stats command, we don't want to include it in the cascade operation query
      // since where commands that use either the MATCH or MATCH_PHRASE function cannot be immediately followed by a stats command
      // and moreover this STATS command doesn't provide any useful context even if it defines new runtime fields
      // as we would have already selected the definition of the field if our operation stats command references it
      return;
    }
    _esqlAst.mutate.generic.commands.append(cascadeOperationQuery.ast, cmd);
  });

  // we select the first argument because that's the field being categorized
  // {@link https://www.elastic.co/docs/reference/query-languages/esql/functions-operators/grouping-functions#esql-categorize | here}
  const categorizedField = categorizeCommandNode.definition.args[0];
  let categorizedFieldName;
  if ((0, _esqlAst.isFunctionExpression)(categorizedField)) {
    // this assumes that the function invoked accepts a column as its first argument and is not in itself another function invocation
    categorizedFieldName = categorizedField.args[0].text;
  } else {
    categorizedFieldName = categorizedField.name;
    let categorizedFieldNameParamValue;
    if (categorizedFieldNameParamValue = getFieldParamDefinition(categorizedFieldName, categorizeCommandNode.terminals, esqlVariables)) {
      if (typeof categorizedFieldNameParamValue === 'string') {
        // we expect the categorized field name parameter value to be a string, so we check for that and update the categorized field name to the param value if it is
        categorizedFieldName = categorizedFieldNameParamValue;
      }
    }
  }
  const matchValue = nodePathMap[removeBackticks(categorizeCommandNode.column.name)];

  // build a where command with match expressions for the selected categorize function
  const categorizeWhereCommand = _esqlAst.Builder.command({
    name: 'where',
    args: [_esqlAst.Builder.expression.func.call('match', [
    // this search doesn't work well on the keyword field when used with the match function, so we remove the keyword suffix to get the actual field name
    _esqlAst.Builder.identifier({
      name: categorizedFieldName.replace(/\.keyword\b/i, '')
    }), _esqlAst.Builder.expression.literal.string((0, _extract_categorize_tokens.extractCategorizeTokens)(matchValue).join(' ')), _esqlAst.Builder.expression.map({
      entries: [_esqlAst.Builder.expression.entry('auto_generate_synonyms_phrase_query', _esqlAst.Builder.expression.literal.boolean(false)), _esqlAst.Builder.expression.entry('fuzziness', _esqlAst.Builder.expression.literal.integer(0)), _esqlAst.Builder.expression.entry('operator', _esqlAst.Builder.expression.literal.string('AND'))]
    })])]
  });
  _esqlAst.mutate.generic.commands.append(cascadeOperationQuery.ast, categorizeWhereCommand);
  return {
    esql: _esqlAst.BasicPrettyPrinter.print(cascadeOperationQuery.ast)
  };
}

/**
 * Modifies the provided ESQL query to only include the specified columns in the stats by option.
 */
function mutateQueryStatsGrouping(query, pick) {
  var _getStatsCommandToOpe;
  const EditorESQLQuery = _esqlAst.EsqlQuery.fromSrc(query.esql);
  const dataSourceCommand = getESQLQueryDataSourceCommand(EditorESQLQuery);
  if (!dataSourceCommand) {
    throw new Error('Query does not have a data source');
  }
  const statsCommands = Array.from(_esqlAst.mutate.commands.stats.list(EditorESQLQuery.ast));
  if (statsCommands.length === 0) {
    throw new Error(`Query does not include a "stats" command`);
  }
  const {
    grouping: statsCommandToOperateOnGrouping,
    command: statsCommandToOperateOn
  } = (_getStatsCommandToOpe = getStatsCommandToOperateOn(EditorESQLQuery)) !== null && _getStatsCommandToOpe !== void 0 ? _getStatsCommandToOpe : {};
  if (!statsCommandToOperateOn) {
    throw new Error(`No valid "stats" command was found in the query`);
  }
  const isValidPick = pick.every(col => Object.keys(statsCommandToOperateOnGrouping).includes(col) || Object.keys(statsCommandToOperateOnGrouping).includes(`\`${col}\``));
  if (!isValidPick) {
    // nothing to do, return query as is
    return {
      esql: _esqlAst.BasicPrettyPrinter.print(EditorESQLQuery.ast)
    };
  }

  // Create a modified stats command with only the specified column as args for the "by" option
  const modifiedStatsCommand = _esqlAst.Builder.command({
    name: 'stats',
    args: statsCommandToOperateOn.args.map(statsCommandArg => {
      if ((0, _esqlAst.isOptionNode)(statsCommandArg) && statsCommandArg.name === 'by') {
        return _esqlAst.Builder.option({
          name: statsCommandArg.name,
          args: statsCommandArg.args.reduce((acc, cur) => {
            var _find$name, _find, _cur$args$find$name, _cur$args$find;
            if ((0, _esqlAst.isColumn)(cur) && pick.includes(removeBackticks(cur.name))) {
              acc.push(_esqlAst.synth.exp(cur.text, {
                withFormatting: false
              }));
            } else if ((0, _esqlAst.isFunctionExpression)(cur) && isSupportedStatsFunction(cur.subtype === 'variadic-call' ? cur.name : (_find$name = (_find = cur.args[1].find(_esqlAst.isFunctionExpression)) === null || _find === void 0 ? void 0 : _find.name) !== null && _find$name !== void 0 ? _find$name : '') && pick.includes(cur.subtype === 'variadic-call' ? cur.text : removeBackticks((_cur$args$find$name = (_cur$args$find = cur.args.find(_esqlAst.isColumn)) === null || _cur$args$find === void 0 ? void 0 : _cur$args$find.name) !== null && _cur$args$find$name !== void 0 ? _cur$args$find$name : ''))) {
              acc.push(_esqlAst.synth.exp(cur.text, {
                withFormatting: false
              }));
            }
            return acc;
          }, [])
        });
      }

      // leverage synth to clone the rest of the args since we'd want to use those parts as is
      return _esqlAst.synth.exp(statsCommandArg.text, {
        withFormatting: false
      });
    })
  });

  // Get the position of the original stats command
  const statsCommandIndex = EditorESQLQuery.ast.commands.findIndex(cmd => cmd.text === statsCommandToOperateOn.text);

  // remove stats command
  _esqlAst.mutate.generic.commands.remove(EditorESQLQuery.ast, statsCommandToOperateOn);

  // insert modified stats command at same position previous one was at
  _esqlAst.mutate.generic.commands.insert(EditorESQLQuery.ast, modifiedStatsCommand, statsCommandIndex);
  return {
    esql: _esqlAst.BasicPrettyPrinter.print(EditorESQLQuery.ast)
  };
}

/**
 * Handles the computation and appending of a filtering where clause,
 * for ES|QL query containing a stats command in the cascade layout experience
 */
const appendFilteringWhereClauseForCascadeLayout = (query, esqlVariables, dataView, fieldName, value, operation, fieldType) => {
  const ESQLQuery = _esqlAst.EsqlQuery.fromSrc(query);

  // we make an initial assumption that the filtering operation's target field was declared by the stats command driving the cascade experience
  let fieldDeclarationCommandSummary = getStatsCommandToOperateOn(ESQLQuery);

  // when the grouping option is an unnamed function, it's wrapped in backticks in the generated AST so we test for that first, else we assume this does not apply
  let normalizedFieldName = fieldDeclarationCommandSummary.grouping[`\`${fieldName}\``] ? `\`${fieldName}\`` : fieldName;
  const isFieldUsedInOperatingStatsCommand = fieldDeclarationCommandSummary.grouping[normalizedFieldName];

  // create placeholder for the insertion anchor command which is the command that is most suited to accept the user's requested filtering operation
  let insertionAnchorCommand;

  // create placeholder for a flag to indicate if the field was declared in a stats command
  let isFieldRuntimeDeclared = false;

  // placeholder for the computed expression of the filter operation that has been requested by the user
  let computedFilteringExpression;
  if (isFieldUsedInOperatingStatsCommand) {
    // if the field name is marked as a new field then we know it was declared by the stats command driving the cascade experience,
    // so we set the flag to true and use the stats command as the insertion anchor command
    if (fieldDeclarationCommandSummary.newFields.has(normalizedFieldName)) {
      isFieldRuntimeDeclared = true;
    } else {
      var _groupDeclarationComm2;
      // otherwise, we need to ascertain that the field was not created by a preceding stats command
      // so we check the runtime fields created by the stats commands in the query to see if the field was declared in one of them
      const statsCommandRuntimeFields = getStatsCommandRuntimeFields(ESQLQuery);

      // attempt to find the index of the stats command that declared the field
      const groupDeclarationCommandIndex = statsCommandRuntimeFields.findIndex(field => field.has(normalizedFieldName));

      // if the field was declared in a stats command, then we set the flag to true
      isFieldRuntimeDeclared = groupDeclarationCommandIndex >= 0;
      let groupDeclarationCommandSummary = null;
      if (isFieldRuntimeDeclared) {
        groupDeclarationCommandSummary = getStatsCommandAtIndexSummary(ESQLQuery, groupDeclarationCommandIndex);
      }
      fieldDeclarationCommandSummary = (_groupDeclarationComm2 = groupDeclarationCommandSummary) !== null && _groupDeclarationComm2 !== void 0 ? _groupDeclarationComm2 : fieldDeclarationCommandSummary;
    }
    insertionAnchorCommand = fieldDeclarationCommandSummary.command;
    let fieldNameParamValue;
    if (fieldNameParamValue = getFieldParamDefinition(fieldName, fieldDeclarationCommandSummary.grouping[normalizedFieldName].terminals, esqlVariables)) {
      if (typeof fieldNameParamValue === 'string') {
        // we expect the field name parameter value to be a string, so we check for that and update the normalized field name to the param value if it is
        normalizedFieldName = fieldNameParamValue;
      }
    }
  } else {
    // if the requested field doesn't exist on the stats command that's driving the cascade experience,
    // then the requested field is a field on the index data given this, we opt to use the data source command as the insertion anchor command
    insertionAnchorCommand = getESQLQueryDataSourceCommand(ESQLQuery);
  }
  const {
    operator,
    expressionType
  } = (0, _utils.getOperator)(operation);

  // if the value being filtered on is not "aggregatable" and is either a text or keyword field, we opt to use match phrase for the where command
  const shouldUseMatchPhrase = requiresMatchPhrase(normalizedFieldName, dataView.fields);
  if (shouldUseMatchPhrase) {
    const matchPhraseExpression = _esqlAst.Builder.expression.func.call('match_phrase', [_esqlAst.Builder.identifier({
      name: removeBackticks(normalizedFieldName)
    }), _esqlAst.Builder.expression.literal.string(value)]);
    computedFilteringExpression = expressionType === 'postfix-unary' ? _esqlAst.Builder.expression.func.postfix(operator, [matchPhraseExpression]) : operator === '!=' ? _esqlAst.Builder.expression.func.unary('not', [matchPhraseExpression]) : matchPhraseExpression;
  } else {
    computedFilteringExpression = expressionType === 'postfix-unary' ? _esqlAst.Builder.expression.func.postfix(operator, [_esqlAst.Builder.identifier({
      name: removeBackticks(normalizedFieldName)
    })]) : _esqlAst.Builder.expression.func.binary(operator, [_esqlAst.Builder.identifier({
      name: removeBackticks(normalizedFieldName)
    }), fieldType && isSupportedFieldType(fieldType) ? fieldType === 'boolean' ? _esqlAst.Builder.expression.literal.boolean(value) : fieldType === 'string' ? _esqlAst.Builder.expression.literal.string(value) : _esqlAst.Builder.expression.literal[fieldType](value) :
    // when fieldType is not provided or supported, we default to string
    Number.isNaN(Number(value)) ? _esqlAst.Builder.expression.literal.string(value) : _esqlAst.Builder.expression.literal.integer(value)]);
  }
  const insertionAnchorCommandIndex = ESQLQuery.ast.commands.findIndex(cmd => cmd.text === insertionAnchorCommand.text);

  // since we can't anticipate the nature of the query we could be dealing with
  // when its a runtime field or not used in the operating stats command, we need to insert the new where command right after the insertion anchor command
  // otherwise we insert it right before the insertion anchor command
  const commandsFollowingInsertionAnchor = !isFieldUsedInOperatingStatsCommand || isFieldRuntimeDeclared ? ESQLQuery.ast.commands.slice(insertionAnchorCommandIndex, insertionAnchorCommandIndex + 2) : ESQLQuery.ast.commands.slice(insertionAnchorCommandIndex - 1, insertionAnchorCommandIndex + 1);

  // we search to see if there already exists a where command that applies to our insertion anchor command
  const filteringWhereCommandIndex = commandsFollowingInsertionAnchor.findIndex(cmd => cmd.name === 'where');
  if (filteringWhereCommandIndex < 0) {
    const filteringWhereCommand = _esqlAst.Builder.command({
      name: 'where',
      args: [computedFilteringExpression]
    });
    _esqlAst.mutate.generic.commands.insert(ESQLQuery.ast, filteringWhereCommand,
    // when the field is a runtime field or not used in the operating stats command, we insert the new where command right after the insertion anchor command
    // otherwise we insert it right before the insertion anchor command
    !isFieldUsedInOperatingStatsCommand || isFieldRuntimeDeclared ? insertionAnchorCommandIndex + 1 : insertionAnchorCommandIndex);
    return _esqlAst.BasicPrettyPrinter.print(ESQLQuery.ast);
  }
  const filteringWhereCommand = commandsFollowingInsertionAnchor[filteringWhereCommandIndex];
  let modifiedFilteringWhereCommand = null;

  // the where command itself represents it's expressions as a single argument that could a series of nested left and right expressions
  // hence why only access the first argument.
  const filteringExpression = filteringWhereCommand.args[0];
  if ((0, _esqlAst.isBinaryExpression)(filteringExpression) && filteringExpression.name === 'and') {
    // This is already a combination of some conditions, for now we'll just append the new condition to the existing one
    modifiedFilteringWhereCommand = _esqlAst.synth.cmd`WHERE ${computedFilteringExpression} AND ${filteringExpression}`;
  } else {
    modifiedFilteringWhereCommand = (0, _esqlAst.isBinaryExpression)(filteringExpression) && filteringExpression.args[0].name === normalizedFieldName ?
    // when the expression is a binary expression and it's left hand's value matches the target field we're trying to filter on, we simply replace it with the new expression
    _esqlAst.synth.cmd`WHERE ${computedFilteringExpression}` : _esqlAst.synth.cmd`WHERE ${computedFilteringExpression} AND ${filteringExpression}`;
  }

  // if we where able to create a new filtering where command, we need to add it in and remove the old one
  if (modifiedFilteringWhereCommand) {
    const insertionIndex = ESQLQuery.ast.commands.findIndex(cmd => cmd.text === filteringWhereCommand.text);
    _esqlAst.mutate.generic.commands.insert(ESQLQuery.ast, modifiedFilteringWhereCommand, insertionIndex);
    _esqlAst.mutate.generic.commands.remove(ESQLQuery.ast, filteringWhereCommand);
  }
  return _esqlAst.BasicPrettyPrinter.print(ESQLQuery.ast);
};
exports.appendFilteringWhereClauseForCascadeLayout = appendFilteringWhereClauseForCascadeLayout;