"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getEsqlColumns = void 0;
var _fieldTypes = require("@kbn/field-types");
var _run_query = require("../run_query");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Gets the columns of an ESQL query, formatted as ESQLFieldWithMetadata
 * @param esqlQuery The ESQL query to execute
 * @param search The search service to use
 * @param variables Optional ESQL control variables to substitute in the query
 * @param signal Optional AbortSignal to cancel the request
 * @param timeRange Optional time range for the query
 * @returns A promise that resolves to an array of ESQLFieldWithMetadata
 */
const getEsqlColumns = async ({
  esqlQuery,
  search,
  variables,
  signal,
  timeRange
}) => {
  if (esqlQuery) {
    try {
      const columns = await (0, _run_query.getESQLQueryColumns)({
        esqlQuery,
        search,
        dropNullColumns: true,
        variables: variables !== null && variables !== void 0 ? variables : [],
        signal,
        timeRange
      });
      return (columns === null || columns === void 0 ? void 0 : columns.map(c => {
        return {
          name: c.name,
          type: c.meta.esType,
          hasConflict: c.meta.type === _fieldTypes.KBN_FIELD_TYPES.CONFLICT,
          userDefined: false
        };
      })) || [];
    } catch (error) {
      // Handle error
      return [];
    }
  }
  return [];
};
exports.getEsqlColumns = getEsqlColumns;