"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.printAst = void 0;
var _treeDump = require("tree-dump");
var _utils = require("../../ast/visitor/utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Options for printing an AST.
 */

/**
 * Prints an AST expression as a tree structure for debugging purposes.
 *
 * @param top - The root AST expression to print
 * @param options - Options for printing the AST
 * @returns A string representation of the AST tree
 *
 * @example
 * ```typescript
 * const ast = parse('FROM index | WHERE field > 10');
 * console.log(printAst(ast));
 * ```
 */
const printAst = (top, options, tab = '') => {
  var _options$depth, _options$limit, _options$printSrc, _options$src;
  const maxDepth = (_options$depth = options === null || options === void 0 ? void 0 : options.depth) !== null && _options$depth !== void 0 ? _options$depth : 1e3;
  let nodesLeft = (_options$limit = options === null || options === void 0 ? void 0 : options.limit) !== null && _options$limit !== void 0 ? _options$limit : 1e5; // Remaining number of nodes to print
  const printSrc = (_options$printSrc = options === null || options === void 0 ? void 0 : options.printSrc) !== null && _options$printSrc !== void 0 ? _options$printSrc : false;
  const src = (_options$src = options === null || options === void 0 ? void 0 : options.src) !== null && _options$src !== void 0 ? _options$src : '';
  const printNode = (node, currentTab, depth) => {
    if (nodesLeft-- <= 0) {
      return '...';
    }
    const type = node.type || 'unknown';
    const childrenTree = [];
    let location = '';
    let name = '';
    let text = '';
    let inlineDetails = '';
    if (!(options !== null && options !== void 0 && options.compact)) {
      var _options$location;
      name = node.name ? ` "${node.name}"` : '';
      location = node.location && ((_options$location = options === null || options === void 0 ? void 0 : options.location) !== null && _options$location !== void 0 ? _options$location : true) ? ` ${node.location.min}-${node.location.max}` : '';
      text = !!(options !== null && options !== void 0 && options.text) && node.text ? `, text = "${node.text}"` : '';
      if (node.incomplete) {
        inlineDetails += ' INCOMPLETE';
      }
    }
    if (maxDepth > depth) {
      for (const child of (0, _utils.childrenOfAnyNode)(node)) {
        childrenTree.push(tabNested => printNode(child, tabNested, depth + 1));
      }
    } else {
      childrenTree.push(() => '...');
    }
    let nodeSrc = '';
    if (printSrc && src && node.location) {
      const {
        min,
        max
      } = node.location;
      if (min >= 0 && max >= min) {
        nodeSrc = ` "${src.slice(min, max + 1)}"`.replace(/\n/g, '\\n');
      }
    }
    const header = `${type}${location}${name}${text}${inlineDetails}${nodeSrc}`;
    return header + (0, _treeDump.printTree)(currentTab, childrenTree);
  };
  return printNode(top, tab, 1);
};
exports.printAst = printAst;