"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getSignatureHelp = getSignatureHelp;
var _parser = require("../../parser");
var _ast = require("../../ast");
var _utils = require("../../commands/definitions/utils");
var _columns_retrieval_helpers = require("../shared/columns_retrieval_helpers");
var _subqueries_helpers = require("../shared/subqueries_helpers");
var _get_query_for_fields = require("../shared/get_query_for_fields");
var _query_syntax_helpers = require("../shared/query_syntax_helpers");
var _helpers = require("./helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const MAX_PARAM_TYPES_TO_SHOW = 3;
async function getSignatureHelp(fullText, offset, callbacks) {
  var _fnDefinition$signatu, _fnDefinition$signatu2;
  const innerText = fullText.substring(0, offset);

  // Corrects the query to be able to work with incomplete syntax
  const correctedQuery = (0, _query_syntax_helpers.correctQuerySyntax)(fullText, offset);
  const {
    root
  } = _parser.Parser.parse(correctedQuery);

  // Find the function node that contains the cursor
  let fnNode;
  _ast.Walker.walk(root, {
    visitFunction: fn => {
      const leftParen = fullText.indexOf('(', fn.location.min);
      if (leftParen < offset && (0, _ast.within)(offset - 1, fn)) {
        fnNode = fn;
      }
    }
  });
  if (!fnNode) {
    return undefined;
  }

  // Get the function definition
  const fnDefinition = (0, _utils.getFunctionDefinition)(fnNode.name);
  if (!fnDefinition) {
    return undefined;
  }

  // Calculate the argument to highlight based on cursor position
  let currentArgIndex = (0, _helpers.getArgumentToHighlightIndex)(innerText, fnNode, offset);
  // Handle repeating signatures like CASE(cond, value, cond, value...).
  const isSignatureRepeating = (_fnDefinition$signatu = (_fnDefinition$signatu2 = fnDefinition.signatures) === null || _fnDefinition$signatu2 === void 0 ? void 0 : _fnDefinition$signatu2.some(sig => sig.isSignatureRepeating)) !== null && _fnDefinition$signatu !== void 0 ? _fnDefinition$signatu : false;
  if (isSignatureRepeating) {
    var _fnDefinition$signatu3, _fnDefinition$signatu4;
    const maxParams = Math.max(...((_fnDefinition$signatu3 = (_fnDefinition$signatu4 = fnDefinition.signatures) === null || _fnDefinition$signatu4 === void 0 ? void 0 : _fnDefinition$signatu4.map(sig => sig.params.length)) !== null && _fnDefinition$signatu3 !== void 0 ? _fnDefinition$signatu3 : [1]));
    currentArgIndex = currentArgIndex % maxParams;
  }

  // Gets the columns map of the query, to do type matching for the function arguments
  const {
    subQuery
  } = (0, _subqueries_helpers.findSubquery)(root, offset);
  const astForContext = subQuery !== null && subQuery !== void 0 ? subQuery : root;
  const {
    getColumnMap
  } = (0, _columns_retrieval_helpers.getColumnsByTypeRetriever)((0, _get_query_for_fields.getQueryForFields)(fullText, astForContext), fullText, callbacks);
  const columnsMap = await getColumnMap();

  // Get the formatted function signature, with type filtering based on current args
  const formattedSignature = (0, _utils.getFormattedFunctionSignature)(fnDefinition, fnNode, columnsMap, MAX_PARAM_TYPES_TO_SHOW);
  const parameters = (0, _helpers.getParameterList)(formattedSignature);
  const signature = {
    label: formattedSignature,
    parameters: parameters.map(param => {
      var _fnDefinition$signatu5;
      const paramDefinition = (_fnDefinition$signatu5 = fnDefinition.signatures) === null || _fnDefinition$signatu5 === void 0 ? void 0 : _fnDefinition$signatu5.flatMap(sig => sig.params).find(p => param.startsWith(p.name));
      return {
        label: param,
        documentation: paramDefinition !== null && paramDefinition !== void 0 && paramDefinition.description ? paramDefinition === null || paramDefinition === void 0 ? void 0 : paramDefinition.description : ''
      };
    }) || []
  };
  return {
    signatures: [signature],
    activeSignature: 0,
    // Math.min for the variadic functions, that can have more arguments than the defined parameters
    activeParameter: Math.min(currentArgIndex, parameters.length - 1)
  };
}