"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.isQualifiedColumnShorthand = exports.holeToFragment = void 0;
var _builder = require("../../ast/builder");
var _pretty_print = require("../../pretty_print");
var _is = require("../../ast/is");
var _synth_node = require("./synth_node");
var _synth_literal_fragment = require("./synth_literal_fragment");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

class UnexpectedSynthHoleError extends Error {
  constructor(hole) {
    super(`Unexpected synth hole: ${JSON.stringify(hole)}`);
  }
}
const isColumnShorthand = hole => {
  return Array.isArray(hole) && hole.every(part => typeof part === 'string');
};

/**
 * Identifies qualified column shorthand tuples.
 * ['qualifier', ['fieldName']]
 * ['qualifier', ['fieldPart1', 'fieldPart2']]
 * @param hole
 * @returns
 */
const isQualifiedColumnShorthand = hole => {
  return Array.isArray(hole) && hole.length === 2 && typeof hole[0] === 'string' && isColumnShorthand(hole[1]);
};

/**
 * Converts a synth template hole to a string fragment. A *hole" in a tagged
 * template is JavaScript value, which is returned as-is, during the template
 * execution:
 *
 * ```js
 * tag `SELECT * FROM table WHERE id = ${hole}`;
 * ```
 *
 * In the above example, the `hole` value is fed into the `tag` function as-is
 * (it is not converted to a string). This function is used to convert the
 * hole to a string fragment, which can be used in the query string.
 *
 * @param hole The hole to convert to a string fragment.
 * @returns A string representation of the hole.
 */
exports.isQualifiedColumnShorthand = isQualifiedColumnShorthand;
const holeToFragment = hole => {
  switch (typeof hole) {
    case 'string':
      {
        const node = _builder.Builder.expression.literal.string(hole);
        return _pretty_print.LeafPrinter.string(node);
      }
    case 'number':
      {
        const isInteger = Math.round(hole) === hole;
        const node = isInteger ? _builder.Builder.expression.literal.integer(hole) : _builder.Builder.expression.literal.decimal(hole);
        return _pretty_print.LeafPrinter.literal(node);
      }
    case 'boolean':
      {
        const node = _builder.Builder.expression.literal.boolean(hole);
        return _pretty_print.LeafPrinter.literal(node);
      }
    case 'object':
      {
        if (hole instanceof _synth_literal_fragment.SynthLiteralFragment) {
          return hole.value;
        }
        if (isColumnShorthand(hole)) {
          const node = _builder.Builder.expression.column(hole);
          return _pretty_print.LeafPrinter.column(node);
        }
        if (isQualifiedColumnShorthand(hole)) {
          const qualifier = hole[0];
          const columnParts = hole[1];
          const node = _builder.Builder.expression.column(columnParts, qualifier);
          return _pretty_print.LeafPrinter.column(node);
        }
        if (Array.isArray(hole)) {
          let list = '';
          for (const item of hole) {
            list += (list ? ', ' : '') + holeToFragment(item);
          }
          return list;
        } else if (hole instanceof _synth_node.SynthNode || (0, _is.isProperNode)(hole)) {
          return _pretty_print.BasicPrettyPrinter.print(hole);
        }
        throw new UnexpectedSynthHoleError(hole);
      }
    default:
      {
        throw new UnexpectedSynthHoleError(hole);
      }
  }
};
exports.holeToFragment = holeToFragment;