"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.autocomplete = autocomplete;
var _esqlTypes = require("@kbn/esql-types");
var _is = require("../../../ast/is");
var _types = require("../types");
var _complete_items = require("../complete_items");
var _helpers = require("../../definitions/utils/autocomplete/helpers");
var _utils = require("../../definitions/utils");
var _functions = require("../../definitions/utils/autocomplete/functions");
var _functions2 = require("../../definitions/utils/functions");
var _types2 = require("../../definitions/types");
var _utils2 = require("./utils");
var _ast = require("../../definitions/utils/ast");
var _expressions = require("../../definitions/utils/expressions");
var _location = require("../../../ast/location");
var _all_operators = require("../../definitions/all_operators");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

async function autocomplete(query, command, callbacks, context, cursorPosition = query.length) {
  if (!(callbacks !== null && callbacks !== void 0 && callbacks.getByType)) {
    return [];
  }
  const isInlineStats = command.name === 'inline stats';
  const columnExists = name => (0, _helpers.columnExists)(name, context);
  const innerText = query.substring(0, cursorPosition);
  const pos = (0, _utils2.getPosition)(command, innerText);

  // Find the function at cursor position for suggestions
  const foundFunction = cursorPosition ? findFunctionForSuggestions(command, cursorPosition) : null;
  let functionParameterContext;
  if (foundFunction && (foundFunction.subtype === 'variadic-call' || foundFunction.subtype === 'binary-expression')) {
    functionParameterContext = buildCustomFilteringContext(command, foundFunction, context);
    const isInBy = isNodeWithinByClause(foundFunction, command);
    const isTimeseriesSource = query.trimStart().toLowerCase().startsWith('ts ');

    // Determine the appropriate location based on context
    let location;
    if (isInBy) {
      // BY clause always uses EVAL location
      location = _types.Location.EVAL;
    } else if (isTimeseriesSource && (0, _functions.isAggFunctionUsedAlready)(command, command.args.length - 1)) {
      // Inside aggregate function with TS source command
      location = _types.Location.STATS_TIMESERIES;
    } else {
      // Regular STATS context
      location = _types.Location.STATS;
    }
    const {
      suggestions: functionsSpecificSuggestions
    } = await (0, _utils.suggestForExpression)({
      query,
      expressionRoot: foundFunction,
      command,
      cursorPosition,
      location,
      context,
      callbacks,
      options: {
        // Pass STATS-specific filtering context to be preserved in recursive calls
        functionParameterContext
      }
    });
    if (functionsSpecificSuggestions.length > 0) {
      return functionsSpecificSuggestions;
    }
  }
  switch (pos) {
    case 'expression_without_assignment':
      {
        var _callbacks$getSuggest;
        const isNewMultipleExpression = /,\s*$/.test(innerText);
        const expressionRoot = isNewMultipleExpression ? undefined // we're in a new expression, but there isn't an AST node for it yet
        : command.args[command.args.length - 1];
        if (Array.isArray(expressionRoot)) {
          return [];
        }
        const expressionSuggestions = await getExpressionSuggestions({
          query,
          command,
          cursorPosition,
          expressionRoot,
          location: _types.Location.STATS,
          context,
          callbacks,
          emptySuggestions: [...(!isNewMultipleExpression && !isInlineStats ? [{
            ..._complete_items.byCompleteItem,
            sortText: 'D'
          }] : []), (0, _complete_items.getNewUserDefinedColumnSuggestion)((callbacks === null || callbacks === void 0 ? void 0 : (_callbacks$getSuggest = callbacks.getSuggestedUserDefinedColumnName) === null || _callbacks$getSuggest === void 0 ? void 0 : _callbacks$getSuggest.call(callbacks)) || '')],
          afterCompleteSuggestions: [_complete_items.whereCompleteItem, _complete_items.byCompleteItem, ...(0, _utils2.getCommaAndPipe)(innerText, expressionRoot, columnExists)],
          suggestColumns: false,
          suggestFunctions: true,
          controlType: _esqlTypes.ESQLVariableType.FUNCTIONS,
          functionParameterContext
        });
        return expressionSuggestions;
      }
    case 'expression_after_assignment':
      {
        const assignment = command.args[command.args.length - 1];
        const expressionRoot = (0, _is.isAssignment)(assignment) ? (0, _expressions.getAssignmentExpressionRoot)(assignment) : undefined;
        const expressionSuggestions = await getExpressionSuggestions({
          query,
          command,
          cursorPosition,
          expressionRoot,
          location: _types.Location.STATS,
          context,
          callbacks,
          emptySuggestions: [],
          afterCompleteSuggestions: [_complete_items.whereCompleteItem, _complete_items.byCompleteItem, ...(0, _utils2.getCommaAndPipe)(innerText, expressionRoot, columnExists)],
          suggestColumns: false,
          suggestFunctions: true,
          controlType: _esqlTypes.ESQLVariableType.FUNCTIONS,
          functionParameterContext
        });
        return expressionSuggestions;
      }
    case 'after_where':
      {
        const whereFn = command.args[command.args.length - 1];
        // TODO do we still need this check?
        const expressionRoot = (0, _ast.isMarkerNode)(whereFn.args[1]) ? undefined : whereFn.args[1];
        if (expressionRoot && !!Array.isArray(expressionRoot)) {
          return [];
        }
        const {
          suggestions,
          computed
        } = await (0, _utils.suggestForExpression)({
          query,
          expressionRoot,
          command,
          cursorPosition,
          location: _types.Location.STATS_WHERE,
          context,
          callbacks,
          options: {
            preferredExpressionType: 'boolean'
          }
        });
        const {
          expressionType,
          isComplete
        } = computed;
        if (expressionType === 'boolean' && isComplete) {
          suggestions.push(_complete_items.pipeCompleteItem, {
            ..._complete_items.commaCompleteItem,
            text: ', '
          }, _complete_items.byCompleteItem);
        }
        return suggestions;
      }
    case 'grouping_expression_after_assignment':
      {
        var _context$histogramBar;
        const byNode = command.args[command.args.length - 1];
        const assignment = byNode.args[byNode.args.length - 1];
        const expressionRoot = (0, _is.isAssignment)(assignment) ? (0, _expressions.getAssignmentExpressionRoot)(assignment) : undefined;
        const ignoredColumns = alreadyUsedColumns(command);
        return getExpressionSuggestions({
          query,
          command,
          cursorPosition,
          expressionRoot,
          location: _types.Location.STATS_BY,
          context,
          callbacks,
          emptySuggestions: [(0, _complete_items.getDateHistogramCompletionItem)((_context$histogramBar = context === null || context === void 0 ? void 0 : context.histogramBarTarget) !== null && _context$histogramBar !== void 0 ? _context$histogramBar : 0)],
          afterCompleteSuggestions: (0, _utils2.getCommaAndPipe)(innerText, expressionRoot, columnExists),
          addSpaceAfterFirstField: false,
          ignoredColumns,
          openSuggestions: true,
          functionParameterContext
        });
      }
    case 'grouping_expression_without_assignment':
      {
        var _callbacks$getSuggest2, _context$histogramBar2;
        let expressionRoot;
        if (!/,\s*$/.test(innerText)) {
          const byNode = command.args[command.args.length - 1];
          expressionRoot = byNode.args[byNode.args.length - 1];
        }
        // guaranteed by the getPosition function, but we check it here for type safety
        if (Array.isArray(expressionRoot)) {
          return [];
        }
        const ignoredColumns = alreadyUsedColumns(command);
        return getExpressionSuggestions({
          query,
          command,
          cursorPosition,
          expressionRoot,
          location: _types.Location.STATS_BY,
          context,
          callbacks,
          emptySuggestions: [(0, _complete_items.getNewUserDefinedColumnSuggestion)((callbacks === null || callbacks === void 0 ? void 0 : (_callbacks$getSuggest2 = callbacks.getSuggestedUserDefinedColumnName) === null || _callbacks$getSuggest2 === void 0 ? void 0 : _callbacks$getSuggest2.call(callbacks)) || ''), (0, _complete_items.getDateHistogramCompletionItem)((_context$histogramBar2 = context === null || context === void 0 ? void 0 : context.histogramBarTarget) !== null && _context$histogramBar2 !== void 0 ? _context$histogramBar2 : 0)],
          afterCompleteSuggestions: (0, _utils2.getCommaAndPipe)(innerText, expressionRoot, columnExists),
          addSpaceAfterFirstField: false,
          ignoredColumns,
          openSuggestions: true,
          functionParameterContext
        });
      }
    default:
      return [];
  }
}

// TODO: Verify if ignoredColumns parameter is redundant since suggestForExpression
// already calculates ignored columns internally via deriveIgnoredColumns()
async function getExpressionSuggestions({
  query,
  command,
  cursorPosition,
  expressionRoot,
  location,
  context,
  callbacks,
  emptySuggestions = [],
  afterCompleteSuggestions = [],
  addSpaceAfterFirstField,
  suggestColumns = true,
  suggestFunctions = true,
  controlType,
  ignoredColumns = [],
  openSuggestions,
  functionParameterContext
}) {
  const suggestions = [];
  const modifiedCallbacks = suggestColumns ? callbacks : {
    ...callbacks,
    getByType: undefined
  };
  const {
    suggestions: expressionSuggestions,
    computed
  } = await (0, _utils.suggestForExpression)({
    query,
    expressionRoot,
    command,
    cursorPosition,
    location,
    context,
    callbacks: modifiedCallbacks,
    options: {
      addSpaceAfterFirstField,
      ignoredColumnsForEmptyExpression: ignoredColumns,
      suggestFields: suggestColumns,
      suggestFunctions,
      controlType,
      openSuggestions,
      functionParameterContext
    }
  });
  const {
    insideFunction
  } = computed;
  if (!(0, _utils2.rightAfterColumn)(computed.innerText, expressionRoot, name => (0, _helpers.columnExists)(name, context))) {
    suggestions.push(...expressionSuggestions);
  }
  if ((!expressionRoot || (0, _is.isColumn)(expressionRoot) && !(0, _helpers.columnExists)(expressionRoot.parts.join('.'), context)) && computed.position !== 'after_not' && !insideFunction) {
    suggestions.push(...emptySuggestions);
  }
  if (computed.isComplete && !insideFunction) {
    suggestions.push(...afterCompleteSuggestions);
  }
  return suggestions;
}
function getByOption(command) {
  return command.args.find(arg => !Array.isArray(arg) && arg.name === 'by');
}
function isNodeWithinByClause(node, command) {
  const byOption = getByOption(command);
  return byOption ? (0, _location.within)(node, byOption) : false;
}
function alreadyUsedColumns(command) {
  var _byOption$args$filter;
  const byOption = getByOption(command);
  const columnNodes = (_byOption$args$filter = byOption === null || byOption === void 0 ? void 0 : byOption.args.filter(arg => !Array.isArray(arg) && arg.type === 'column')) !== null && _byOption$args$filter !== void 0 ? _byOption$args$filter : [];
  return columnNodes.map(node => node.parts.join('.'));
}

// Builds function filtering context: always ignore grouping functions,
// in main STATS clause also filter conflicting aggregate functions
function buildCustomFilteringContext(command, foundFunction, context) {
  if (!foundFunction) {
    return undefined;
  }
  const basicContext = (0, _utils.buildExpressionFunctionParameterContext)(foundFunction, context);
  if (!basicContext) {
    return undefined;
  }
  const statsSpecificFunctionsToIgnore = [];
  // Always ignore grouping functions in all contexts
  statsSpecificFunctionsToIgnore.push(...(0, _functions2.getAllFunctions)({
    type: _types2.FunctionDefinitionTypes.GROUPING
  }).map(({
    name
  }) => name));
  const finalCommandArgIndex = command.args.length - 1;
  const isInBy = isNodeWithinByClause(foundFunction, command);
  if (!isInBy) {
    statsSpecificFunctionsToIgnore.push(...(0, _functions.getFunctionsToIgnoreForStats)(command, finalCommandArgIndex), ...((0, _functions.isAggFunctionUsedAlready)(command, finalCommandArgIndex) ? (0, _functions2.getAllFunctions)({
      type: _types2.FunctionDefinitionTypes.AGG
    }).map(({
      name
    }) => name) : []), ...((0, _functions.isTimeseriesAggUsedAlready)(command, finalCommandArgIndex) ? (0, _functions2.getAllFunctions)({
      type: _types2.FunctionDefinitionTypes.TIME_SERIES_AGG
    }).map(({
      name
    }) => name) : []));
  }
  return {
    ...basicContext,
    functionsToIgnore: [...basicContext.functionsToIgnore, ...statsSpecificFunctionsToIgnore]
  };
}
function findFunctionForSuggestions(command, cursorPosition) {
  const {
    node,
    containingFunction
  } = (0, _ast.findAstPosition)({
    type: 'query',
    commands: [command]
  }, cursorPosition);
  if (node && node.type === 'function') {
    const fn = node;
    const isSpecialOperator = _all_operators.inOperators.some(op => op.name === fn.name) || _all_operators.nullCheckOperators.some(op => op.name === fn.name);
    if (fn.subtype === 'variadic-call' || fn.subtype === 'binary-expression' && isSpecialOperator) {
      return fn;
    }
  }
  return containingFunction || null;
}