"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.CaretPosition = void 0;
exports.getPosition = getPosition;
exports.isAfterPotentialTargetFieldWithSpace = isAfterPotentialTargetFieldWithSpace;
var _is = require("../../../ast/is");
var _expressions = require("../../definitions/utils/expressions");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */
let CaretPosition = exports.CaretPosition = /*#__PURE__*/function (CaretPosition) {
  CaretPosition[CaretPosition["RERANK_KEYWORD"] = 0] = "RERANK_KEYWORD";
  CaretPosition[CaretPosition["RERANK_AFTER_TARGET_FIELD"] = 1] = "RERANK_AFTER_TARGET_FIELD";
  CaretPosition[CaretPosition["RERANK_AFTER_TARGET_ASSIGNMENT"] = 2] = "RERANK_AFTER_TARGET_ASSIGNMENT";
  CaretPosition[CaretPosition["ON_KEYWORD"] = 3] = "ON_KEYWORD";
  CaretPosition[CaretPosition["ON_WITHIN_FIELD_LIST"] = 4] = "ON_WITHIN_FIELD_LIST";
  CaretPosition[CaretPosition["ON_KEEP_SUGGESTIONS_AFTER_TRAILING_SPACE"] = 5] = "ON_KEEP_SUGGESTIONS_AFTER_TRAILING_SPACE";
  CaretPosition[CaretPosition["ON_EXPRESSION"] = 6] = "ON_EXPRESSION";
  CaretPosition[CaretPosition["AFTER_WITH_KEYWORD"] = 7] = "AFTER_WITH_KEYWORD";
  CaretPosition[CaretPosition["WITHIN_MAP_EXPRESSION"] = 8] = "WITHIN_MAP_EXPRESSION";
  CaretPosition[CaretPosition["AFTER_COMMAND"] = 9] = "AFTER_COMMAND";
  return CaretPosition;
}({}); // Command is complete, suggest pipe
/**
 * Determines caret position in RERANK command
 */
function getPosition(query, command) {
  const rerankCommand = command;
  const innerText = query.substring(rerankCommand.location.min);
  const onMap = rerankCommand.args[1];
  const withArg = rerankCommand.args[2];
  if (withArg && 'type' in withArg && withArg.type === 'option') {
    const withMap = withArg.args[0];
    if (!withMap || withMap.incomplete && !withMap.text) {
      return {
        position: CaretPosition.AFTER_WITH_KEYWORD
      };
    }
    if (withMap.text && withMap.incomplete) {
      return {
        position: CaretPosition.WITHIN_MAP_EXPRESSION
      };
    }
    if (!withMap.incomplete) {
      return {
        position: CaretPosition.AFTER_COMMAND
      };
    }
  }
  if (onMap) {
    var _rerankCommand$fields;
    const lastField = (_rerankCommand$fields = rerankCommand.fields) === null || _rerankCommand$fields === void 0 ? void 0 : _rerankCommand$fields[rerankCommand.fields.length - 1];
    if (lastField && (0, _is.isAssignment)(lastField)) {
      return {
        position: CaretPosition.ON_EXPRESSION,
        context: {
          expressionRoot: (0, _expressions.getAssignmentExpressionRoot)(lastField)
        }
      };
    }
    if (!lastField.incomplete && isAfterCompleteFieldWithSpace(innerText)) {
      return {
        position: CaretPosition.ON_KEEP_SUGGESTIONS_AFTER_TRAILING_SPACE
      };
    }
    return {
      position: CaretPosition.ON_WITHIN_FIELD_LIST
    };
  }
  if (!!(rerankCommand.query && !rerankCommand.query.incomplete)) {
    return {
      position: CaretPosition.ON_KEYWORD
    };
  }

  // Check targetField (only if query is not complete)
  if (rerankCommand.targetField && !rerankCommand.targetField.incomplete) {
    return {
      position: CaretPosition.RERANK_AFTER_TARGET_ASSIGNMENT
    };
  }
  if (isAfterPotentialTargetFieldWithSpace(innerText)) {
    return {
      position: CaretPosition.RERANK_AFTER_TARGET_FIELD
    };
  }
  return {
    position: CaretPosition.RERANK_KEYWORD
  };
}
function isAfterPotentialTargetFieldWithSpace(innerText) {
  // Pattern: must have at least one non-quote word followed by space
  // !commandText.includes('"') is covered by 'withinQuotes' function in autocomplete.ts
  // However, we keep this extra safety check if it will be accidentally removed
  return innerText.endsWith(' ') && !innerText.includes('"') && /rerank\s+\w+\s+$/i.test(innerText);
}
function isAfterCompleteFieldWithSpace(innerText) {
  // Matches "ON" followed by any content ending with non-comma/non-space character + space
  const completeWithSpace = /\bon\s+.*[^,\s]\s+$/i;
  const endsWithCommaSpace = /,\s+$/i;
  return completeWithSpace.test(innerText) && !endsWithCommaSpace.test(innerText);
}