"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.validate = void 0;
var _is = require("../../../ast/is");
var _errors = require("../../definitions/utils/errors");
var _validation = require("../../definitions/utils/validation");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const validate = (command, ast, context, callbacks) => {
  const messages = [];
  const {
    commandType,
    args
  } = command;
  const joinSources = (context === null || context === void 0 ? void 0 : context.joinSources) || [];
  if (!['left', 'right', 'lookup'].includes(commandType)) {
    return [_errors.errors.unexpected(command.location, 'JOIN command type')];
  }
  const target = args[0];
  let index;
  let alias;
  if ((0, _is.isBinaryExpression)(target)) {
    if (target.name === 'as') {
      alias = target.args[1];
      index = target.args[0];
      if (!(0, _is.isSource)(index) || !(0, _is.isIdentifier)(alias)) {
        return [_errors.errors.unexpected(target.location)];
      }
    } else {
      return [_errors.errors.unexpected(target.location)];
    }
  } else if ((0, _is.isSource)(target)) {
    index = target;
  } else {
    return [_errors.errors.unexpected(target.location)];
  }
  let isIndexFound = false;
  for (const {
    name,
    aliases
  } of joinSources) {
    if (index.name === name) {
      isIndexFound = true;
      break;
    }
    if (aliases) {
      for (const aliasName of aliases) {
        if (index.name === aliasName) {
          isIndexFound = true;
          break;
        }
      }
    }
  }
  if (!isIndexFound) {
    const error = _errors.errors.invalidJoinIndex(index);
    messages.push(error);
    return messages;
  }

  // Validate JOIN ON expressions
  const onOption = (0, _utils.getOnOption)(command);
  if (onOption) {
    messages.push(...validateOnExpressions(command));
  }
  messages.push(...(0, _validation.validateCommandArguments)(command, ast, context, callbacks));
  return messages;
};
exports.validate = validate;
const validateOnExpressions = joinCommand => {
  const messages = [];
  const onOption = (0, _utils.getOnOption)(joinCommand);
  const expressions = onOption.args;

  // Find complete binary expressions
  const binaryExpressions = expressions.filter(expr => (0, _is.isBinaryExpression)(expr) && !expr.incomplete);

  // Binary expressions cannot be mixed with comma-separated fields
  if (binaryExpressions.length > 0 && expressions.length > 1) {
    messages.push(_errors.errors.joinOnSingleExpression(onOption.location));
  }
  return messages;
};