"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.validate = void 0;
var _errors = require("../../definitions/utils/errors");
var _is = require("../../../ast/is");
var _validation = require("../../definitions/utils/validation");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const validateColumnForGrokDissect = (command, context) => {
  const acceptedColumnTypes = ['keyword', 'text'];
  const astCol = command.args[0];
  const columnRef = context === null || context === void 0 ? void 0 : context.columns.get(astCol.name);
  if (columnRef && !acceptedColumnTypes.includes(columnRef.type)) {
    return [(0, _errors.getMessageFromId)({
      messageId: 'unsupportedColumnTypeForCommand',
      values: {
        command: command.name.toUpperCase(),
        type: acceptedColumnTypes.join(', '),
        givenType: columnRef.type,
        column: astCol.name
      },
      locations: astCol.location
    })];
  }
  return [];
};
const validate = (command, ast, context, callbacks) => {
  const messages = validateColumnForGrokDissect(command, context);
  const appendSeparatorClause = command.args.find(arg => (0, _is.isOptionNode)(arg));
  if (!appendSeparatorClause) {
    return messages;
  }
  if (appendSeparatorClause.name !== 'append_separator') {
    messages.push((0, _errors.getMessageFromId)({
      messageId: 'unknownDissectKeyword',
      values: {
        keyword: appendSeparatorClause.name
      },
      locations: appendSeparatorClause.location
    }));
    return messages;
  }
  const [firstArg] = appendSeparatorClause.args;
  if (!Array.isArray(firstArg) && (!(0, _is.isLiteral)(firstArg) || firstArg.literalType !== 'keyword')) {
    var _ref;
    const value = 'value' in firstArg && !(0, _is.isInlineCast)(firstArg) ? firstArg.value : firstArg.name;
    messages.push((0, _errors.getMessageFromId)({
      messageId: 'wrongDissectOptionArgumentType',
      values: {
        value: (_ref = value) !== null && _ref !== void 0 ? _ref : ''
      },
      locations: firstArg.location
    }));
  }
  messages.push(...(0, _validation.validateCommandArguments)(command, ast, context, callbacks));
  return messages;
};
exports.validate = validate;