"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.validateFunction = validateFunction;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _ = require("..");
var _2 = require("../../../../..");
var _is = require("../../../../ast/is");
var _location = require("../../../registry/location");
var _expressions = require("../expressions");
var _column = require("./column");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

function validateFunction({
  fn,
  parentCommand,
  ast,
  context,
  callbacks
}) {
  const validator = new FunctionValidator(fn, parentCommand, ast, context, callbacks);
  validator.validate();
  return validator.messages;
}
class FunctionValidator {
  constructor(fn, parentCommand, ast, context, callbacks, parentAggFunction = undefined) {
    (0, _defineProperty2.default)(this, "definition", void 0);
    (0, _defineProperty2.default)(this, "argTypes", []);
    (0, _defineProperty2.default)(this, "argLiteralsMask", []);
    (0, _defineProperty2.default)(this, "_messages", []);
    this.fn = fn;
    this.parentCommand = parentCommand;
    this.ast = ast;
    this.context = context;
    this.callbacks = callbacks;
    this.parentAggFunction = parentAggFunction;
    this.definition = (0, _.getFunctionDefinition)(fn.name);
    for (const _arg of this.fn.args) {
      const arg = Array.isArray(_arg) ? _arg[0] : _arg; // for some reason, some args are wrapped in an array, for example named params

      this.argTypes.push((0, _expressions.getExpressionType)(arg, this.context.columns));
      this.argLiteralsMask.push((0, _is.isLiteral)(arg));
    }
  }

  /**
   * Runs validation checks on the function. Once validation is complete,
   * any errors found will be available in this.messages.
   */
  validate() {
    if (this.definition && !this.licenseOk(this.definition.license)) {
      this.report(_.errors.licenseRequired(this.fn, this.definition.license));
    }
    const nestedErrors = this.validateNestedFunctions();

    // if one or more child functions produced errors, report and stop validation
    if (nestedErrors.length) {
      this.report(...nestedErrors);
      return;
    }

    // skip validation for functions with names defined by a parameter
    // e.g. "... | EVAL ??param(..args)"
    if ((0, _is.isParamLiteral)(this.fn.operator)) {
      return;
    }
    if (!this.definition) {
      this.report(_.errors.unknownFunction(this.fn));
      return;
    }
    if (this.parentAggFunction && this.definition.type === _2.FunctionDefinitionTypes.AGG) {
      this.report(_.errors.nestedAggFunction(this.fn, this.parentAggFunction));
      return;
    }
    if (!this.allowedHere) {
      this.report(_.errors.functionNotAllowedHere(this.fn, this.location.displayName));
    }
    if (!this.hasValidArity) {
      this.report(_.errors.wrongNumberArgs(this.fn, this.definition));
      return;
    }
    this.validateArguments();
  }

  /**
   * Validates the function arguments against the function definition
   */
  validateArguments() {
    if (!this.definition) {
      return;
    }
    const S = (0, _expressions.getMatchingSignatures)(this.definition.signatures, this.argTypes, this.argLiteralsMask, true);
    if (!S.length) {
      this.report(_.errors.noMatchingCallSignature(this.fn, this.definition, this.argTypes));
      return;
    }
    if (this.licenseOk(this.definition.license) && !S.some(sig => this.licenseOk(sig.license))) {
      // The function itself is allowed at this license level, but none of the matching signatures are
      this.report(_.errors.licenseRequiredForSignature(this.fn, S[0]));
    }

    // Validate column arguments
    const columnsToValidate = [];
    const flatArgs = this.fn.args.flat();
    for (let i = 0; i < flatArgs.length; i++) {
      const arg = flatArgs[i];
      if (((0, _is.isColumn)(arg) || (0, _is.isIdentifier)(arg)) && !(this.definition.name === '=' && i === 0) &&
      // don't validate left-hand side of assignment
      !(this.definition.name === 'as' && i === 1) // don't validate right-hand side of AS
      ) {
        columnsToValidate.push(arg);
      }
    }
    const columnMessages = columnsToValidate.flatMap(arg => {
      return new _column.ColumnValidator(arg, this.context, this.parentCommand.name).validate();
    });
    this.report(...columnMessages);
  }

  /**
   * Reports one or more validation messages
   */
  report(...messages) {
    this._messages.push(...messages);
  }
  get messages() {
    return this._messages;
  }

  /**
   * Validates the nested functions within the current function
   */
  validateNestedFunctions() {
    var _this$definition, _this$definition2;
    const nestedErrors = [];
    const parentAggFunction = this.parentAggFunction ? this.parentAggFunction : ((_this$definition = this.definition) === null || _this$definition === void 0 ? void 0 : _this$definition.type) === _2.FunctionDefinitionTypes.AGG ? (_this$definition2 = this.definition) === null || _this$definition2 === void 0 ? void 0 : _this$definition2.name : undefined;
    for (const _arg of this.fn.args.flat()) {
      const arg = removeInlineCasts(_arg);
      if ((0, _is.isFunctionExpression)(arg)) {
        const validator = new FunctionValidator(arg, this.parentCommand, this.ast, this.context, this.callbacks, parentAggFunction);
        validator.validate();
        nestedErrors.push(...validator.messages);
      }
      if (nestedErrors.length) {
        return nestedErrors;
      }
    }
    return nestedErrors;
  }

  /**
   * Checks if the current license level is sufficient the given license requirement
   */
  licenseOk(license) {
    const hasMinimumLicenseRequired = this.callbacks.hasMinimumLicenseRequired;
    if (hasMinimumLicenseRequired && license) {
      return hasMinimumLicenseRequired(license);
    }
    return true;
  }

  /**
   * Checks if the function is available in the current context
   */
  get allowedHere() {
    var _this$definition$loca, _this$definition3;
    return (_this$definition$loca = (_this$definition3 = this.definition) === null || _this$definition3 === void 0 ? void 0 : _this$definition3.locationsAvailable.includes(this.location.id)) !== null && _this$definition$loca !== void 0 ? _this$definition$loca : false;
  }

  /**
   * Gets information about the location of the current function
   */
  get location() {
    return (0, _location.getLocationInfo)(this.fn, this.parentCommand, this.ast, !!this.parentAggFunction);
  }

  /**
   * Checks if the function has a valid number of arguments
   */
  get hasValidArity() {
    const {
      min,
      max
    } = getMaxMinNumberOfParams(this.definition);
    const arity = this.fn.args.length;
    return arity >= min && arity <= max;
  }
}

/**
 * Returns the maximum and minimum number of parameters allowed by a function
 *
 * Used for too-many, too-few arguments validation
 */
function getMaxMinNumberOfParams(definition) {
  if (definition.signatures.length === 0) {
    return {
      min: 0,
      max: 0
    };
  }
  let min = Infinity;
  let max = 0;
  definition.signatures.forEach(({
    params,
    minParams
  }) => {
    min = Math.min(min, minParams !== null && minParams !== void 0 ? minParams : params.filter(({
      optional
    }) => !optional).length);
    max = Math.max(max, minParams ? Infinity : params.length);
  });
  return {
    min,
    max
  };
}
function removeInlineCasts(arg) {
  if ((0, _is.isInlineCast)(arg)) {
    return removeInlineCasts(arg.value);
  }
  return arg;
}