"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.additionalSourcesSuggestions = additionalSourcesSuggestions;
exports.buildSourcesDefinitions = void 0;
exports.getSourceSuggestions = getSourceSuggestions;
exports.getSourcesFromCommands = getSourcesFromCommands;
exports.shouldBeQuotedSource = shouldBeQuotedSource;
exports.sourceExists = sourceExists;
exports.specialIndicesToSuggestions = void 0;
var _esqlTypes = require("@kbn/esql-types");
var _i18n = require("@kbn/i18n");
var _helpers = require("./autocomplete/helpers");
var _complete_items = require("../../registry/complete_items");
var _constants = require("../constants");
var _metadata = require("../../registry/options/metadata");
var _shared = require("./shared");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const removeSourceNameQuotes = sourceName => sourceName.startsWith('"') && sourceName.endsWith('"') ? sourceName.slice(1, -1) : sourceName;

// Function to clean a single index string from failure stores
const cleanIndex = inputIndex => {
  let cleaned = inputIndex.trim();

  // Remove '::data' suffix
  if (cleaned.endsWith('::data')) {
    cleaned = cleaned.slice(0, -6);
  }
  // Remove '::failures' suffix
  if (cleaned.endsWith('::failures')) {
    cleaned = cleaned.slice(0, -10);
  }
  return cleaned;
};
function getQuotedText(text) {
  return text.startsWith(`"`) && text.endsWith(`"`) ? text : `"${text}"`;
}
function shouldBeQuotedSource(text) {
  // Based on lexer `fragment UNQUOTED_SOURCE_PART`
  return /[:"=|,[\]\/ \t\r\n]/.test(text);
}
function getSafeInsertSourceText(text) {
  return shouldBeQuotedSource(text) ? getQuotedText(text) : text;
}
const buildSourcesDefinitions = (sources, queryString) => sources.map(({
  name,
  isIntegration,
  title,
  type
}) => {
  let text = getSafeInsertSourceText(name);
  const isTimeseries = type === _esqlTypes.SOURCES_TYPES.TIMESERIES;
  let rangeToReplace;

  // If this is a timeseries source we should replace FROM with TS
  // With TS users can benefit from the timeseries optimizations
  if (isTimeseries && queryString) {
    text = `TS ${text}`;
    rangeToReplace = {
      start: 0,
      end: queryString.length + 1
    };
  }
  return (0, _helpers.withAutoSuggest)({
    label: title !== null && title !== void 0 ? title : name,
    text,
    asSnippet: isIntegration,
    kind: isIntegration ? 'Class' : 'Issue',
    detail: isIntegration ? _i18n.i18n.translate('kbn-esql-ast.esql.autocomplete.integrationDefinition', {
      defaultMessage: _esqlTypes.SOURCES_TYPES.INTEGRATION
    }) : _i18n.i18n.translate('kbn-esql-ast.esql.autocomplete.sourceDefinition', {
      defaultMessage: '{type}',
      values: {
        type: type !== null && type !== void 0 ? type : _esqlTypes.SOURCES_TYPES.INDEX
      }
    }),
    sortText: 'A',
    // with filterText we are explicitly telling the Monaco editor's filtering engine
    //  to display the item when the text FROM  is present in the editor at the specified range,
    // even though the label is different.
    ...(rangeToReplace && {
      rangeToReplace,
      filterText: queryString
    })
  });
});

/**
 * Checks if the source exists in the provided sources set.
 * It supports both exact matches and fuzzy searches.
 *
 * @param index - The index to check, which can be a single value or a comma-separated list.
 * @param sources - A Set of source names to check against.
 * @returns true if the source exists, false otherwise.
 */

// The comma-separated index and the ::data or ::failures suffixes solution is temporary
// till we fix the AST for the quoted index names https://github.com/elastic/kibana/issues/222505.
exports.buildSourcesDefinitions = buildSourcesDefinitions;
function sourceExists(index, sources) {
  if (index.startsWith('-')) {
    return true;
  }
  // Split the index by comma to handle multiple values and clean each part
  const individualIndices = index.split(',').map(item => cleanIndex(item));
  // Check if all individual indices exist in sources
  const allExist = individualIndices.every(singleIndex => {
    // First, check for exact match after removing source name quotes
    if (sources.has(removeSourceNameQuotes(singleIndex))) {
      return true;
    }
    // If not an exact match, perform a fuzzy search
    return Boolean((0, _shared.fuzzySearch)(singleIndex, sources.keys()));
  });
  return allExist;
}
function getSourcesFromCommands(commands, sourceType) {
  var _sourceCommand$args;
  const sourceCommand = commands.find(({
    name
  }) => name === 'from' || name === 'ts');
  const args = (_sourceCommand$args = sourceCommand === null || sourceCommand === void 0 ? void 0 : sourceCommand.args) !== null && _sourceCommand$args !== void 0 ? _sourceCommand$args : [];
  // the marker gets added in queries like "FROM "
  return args.filter(arg => arg.sourceType === sourceType && arg.name !== '' && arg.name !== _constants.EDITOR_MARKER);
}
function getSourceSuggestions(sources, alreadyUsed, queryString) {
  // hide indexes that start with .
  return buildSourcesDefinitions(sources.filter(({
    hidden,
    name
  }) => !hidden && !alreadyUsed.includes(name)).map(({
    name,
    dataStreams,
    title,
    type
  }) => {
    return {
      name,
      isIntegration: Boolean(dataStreams && dataStreams.length),
      title,
      type
    };
  }), queryString);
}
async function additionalSourcesSuggestions(queryText, sources, ignored, recommendedQuerySuggestions) {
  const suggestionsToAdd = await (0, _helpers.handleFragment)(queryText, fragment => sourceExists(fragment, new Set(sources.map(({
    name: sourceName
  }) => sourceName))), (_fragment, rangeToReplace) => {
    return getSourceSuggestions(sources, ignored).map(suggestion => ({
      ...suggestion,
      rangeToReplace
    }));
  }, (fragment, rangeToReplace) => {
    const exactMatch = sources.find(({
      name: _name
    }) => _name === fragment);
    if (exactMatch !== null && exactMatch !== void 0 && exactMatch.dataStreams) {
      // this is an integration name, suggest the datastreams
      const definitions = buildSourcesDefinitions(exactMatch.dataStreams.map(({
        name
      }) => ({
        name,
        isIntegration: false
      })));
      return definitions;
    } else {
      const _suggestions = [(0, _helpers.withAutoSuggest)({
        ..._complete_items.pipeCompleteItem,
        filterText: fragment,
        text: fragment + ' | ',
        rangeToReplace,
        sortText: '0'
      }), (0, _helpers.withAutoSuggest)({
        ..._complete_items.commaCompleteItem,
        filterText: fragment,
        text: fragment + ', ',
        rangeToReplace
      }), {
        ..._metadata.metadataSuggestion,
        filterText: fragment,
        text: fragment + ' METADATA ',
        rangeToReplace
      }, ...recommendedQuerySuggestions.map(suggestion => ({
        ...suggestion,
        rangeToReplace,
        filterText: fragment,
        text: fragment + suggestion.text
      }))];
      return _suggestions;
    }
  });
  return suggestionsToAdd;
}

// Treating lookup and time_series mode indices
const specialIndicesToSuggestions = indices => {
  const mainSuggestions = [];
  const aliasSuggestions = [];
  for (const index of indices) {
    mainSuggestions.push((0, _helpers.withAutoSuggest)({
      label: index.name,
      text: index.name + ' ',
      kind: 'Issue',
      detail: _i18n.i18n.translate('kbn-esql-ast.esql.autocomplete.specialIndexes.indexType.index', {
        defaultMessage: 'Index'
      }),
      sortText: '0-INDEX-' + index.name
    }));
    if (index.aliases) {
      for (const alias of index.aliases) {
        aliasSuggestions.push((0, _helpers.withAutoSuggest)({
          label: alias,
          text: alias + ' $0',
          asSnippet: true,
          kind: 'Issue',
          detail: _i18n.i18n.translate('kbn-esql-ast.esql.autocomplete.specialIndexes.indexType.alias', {
            defaultMessage: 'Alias'
          }),
          sortText: '1-ALIAS-' + alias
        }));
      }
    }
  }
  return [...mainSuggestions, ...aliasSuggestions];
};
exports.specialIndicesToSuggestions = specialIndicesToSuggestions;