"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.fuzzySearch = fuzzySearch;
exports.getColumnByName = getColumnByName;
exports.getColumnForASTNode = getColumnForASTNode;
exports.getOverlapRange = getOverlapRange;
exports.isParamExpressionType = void 0;
exports.isRestartingExpression = isRestartingExpression;
exports.pipePrecedesCurrentWord = pipePrecedesCurrentWord;
exports.removeFinalUnknownIdentiferArg = removeFinalUnknownIdentiferArg;
exports.unescapeColumnName = unescapeColumnName;
var _helpers = require("./autocomplete/helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * In several cases we don't want to count the last arg if it is
 * of type unknown.
 *
 * this solves for the case where the user has typed a
 * prefix (e.g. "keywordField != tex/")
 *
 * "tex" is not a recognizable identifier so it is of
 * type "unknown" which leads us to continue suggesting
 * fields/functions.
 *
 * Monaco will then filter our suggestions list
 * based on the "tex" prefix which gives the correct UX
 */
function removeFinalUnknownIdentiferArg(args, getExpressionType) {
  return getExpressionType(args[args.length - 1]) === 'unknown' ? args.slice(0, args.length - 1) : args;
}

/**
 * Checks the suggestion text for overlap with the current query.
 *
 * This is useful to determine the range of the existing query that should be
 * replaced if the suggestion is accepted.
 *
 * For example
 * QUERY: FROM source | WHERE field IS NO
 * SUGGESTION: IS NOT NULL
 *
 * The overlap is "IS NO" and the range to replace is "IS NO" in the query.
 *
 * @param query
 * @param suggestionText
 * @returns
 */
function getOverlapRange(query, suggestionText) {
  let overlapLength = 0;

  // Convert both strings to lowercase for case-insensitive comparison
  const lowerQuery = query.toLowerCase();
  const lowerSuggestionText = suggestionText.toLowerCase();
  for (let i = 0; i <= lowerSuggestionText.length; i++) {
    const substr = lowerSuggestionText.substring(0, i);
    if (lowerQuery.endsWith(substr)) {
      overlapLength = i;
    }
  }
  if (overlapLength === 0) {
    return;
  }
  return {
    start: query.length - overlapLength,
    end: query.length
  };
}

/**
 * Works backward from the cursor position to determine if
 * the final character of the previous word matches the given character.
 */
function characterPrecedesCurrentWord(text, char) {
  let inCurrentWord = true;
  for (let i = text.length - 1; i >= 0; i--) {
    if (inCurrentWord && /\s/.test(text[i])) {
      inCurrentWord = false;
    }
    if (!inCurrentWord && !/\s/.test(text[i])) {
      return text[i] === char;
    }
  }
}
function pipePrecedesCurrentWord(text) {
  return characterPrecedesCurrentWord(text, '|');
}

/**
 * Are we after a comma? i.e. STATS fieldA, <here>
 */
function isRestartingExpression(text) {
  return (0, _helpers.getLastNonWhitespaceChar)(text) === ',' || characterPrecedesCurrentWord(text, ',');
}

/**
 * Take a column name like "`my``column`"" and return "my`column"
 */
function unescapeColumnName(columnName) {
  // TODO this doesn't cover all escaping scenarios... the best thing to do would be
  // to use the AST column node parts array, but in some cases the AST node isn't available.
  if (columnName.startsWith('`') && columnName.endsWith('`')) {
    return columnName.slice(1, -1).replace(/``/g, '`');
  }
  return columnName;
}

/**
 * This function returns the userDefinedColumn or field matching a column
 */
function getColumnByName(columnName, {
  columns
}) {
  const unescaped = unescapeColumnName(columnName);
  return columns.get(unescaped);
}

/**
 * This function returns the userDefinedColumn or field matching a column
 */
function getColumnForASTNode(node, {
  columns
}) {
  const formatted = node.type === 'identifier' ? node.name : node.parts.join('.');
  return getColumnByName(formatted, {
    columns
  });
}
function hasWildcard(name) {
  return /\*/.test(name);
}
function getMatcher(name, position) {
  if (position === 'start') {
    const prefix = name.substring(1);
    return resource => resource.endsWith(prefix);
  }
  if (position === 'end') {
    const prefix = name.substring(0, name.length - 1);
    return resource => resource.startsWith(prefix);
  }
  if (position === 'multiple-within') {
    // make sure to remove the * at the beginning of the name if present
    const safeName = name.startsWith('*') ? name.slice(1) : name;
    // replace 2 ore more consecutive wildcards with a single one
    const setOfChars = safeName.replace(/\*{2+}/g, '*').split('*');
    return resource => {
      let index = -1;
      return setOfChars.every(char => {
        index = resource.indexOf(char, index + 1);
        return index !== -1;
      });
    };
  }
  const [prefix, postFix] = name.split('*');
  return resource => resource.startsWith(prefix) && resource.endsWith(postFix);
}
function getWildcardPosition(name) {
  if (!hasWildcard(name)) {
    return 'none';
  }
  const wildCardCount = name.match(/\*/g).length;
  if (wildCardCount > 1) {
    return 'multiple-within';
  }
  if (name.startsWith('*')) {
    return 'start';
  }
  if (name.endsWith('*')) {
    return 'end';
  }
  return 'middle';
}

/**
 * Type guard to check if the type is 'param'
 */
const isParamExpressionType = type => type === 'param';
exports.isParamExpressionType = isParamExpressionType;
function fuzzySearch(fuzzyName, resources) {
  const wildCardPosition = getWildcardPosition(fuzzyName);
  if (wildCardPosition !== 'none') {
    const matcher = getMatcher(fuzzyName, wildCardPosition);
    for (const resourceName of resources) {
      if (matcher(resourceName)) {
        return true;
      }
    }
  }
}