"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.suggestAfterOperator = suggestAfterOperator;
var _complete_items = require("../../../../../registry/complete_items");
var _types = require("../../../../types");
var _signature_analyzer = require("../signature_analyzer");
var _expressions = require("../../../expressions");
var _functions = require("../../../functions");
var _shared = require("../../../shared");
var _all_operators = require("../../../../all_operators");
var _dispatcher = require("../operators/dispatcher");
var _is = require("../../../../../../ast/is");
var _suggestion_builder = require("../suggestion_builder");
var _should_suggest_operators = require("./after_complete/should_suggest_operators");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Suggests completions after an operator (e.g., field = |, field IN |)
 * Handles special cases (IN, IS NULL) or delegates to generic operator logic
 */
async function suggestAfterOperator(ctx) {
  const {
    expressionRoot,
    context
  } = ctx;
  if (!expressionRoot) {
    return [];
  }
  const rightmostOperator = getRightmostOperatorInFunctionTree(expressionRoot);
  // If we don't pass rightmostOperator, for "field IN (x) AND field NOT IN (y"
  // dispatchOperators sees AND (no handler) instead of NOT IN, failing to suggest comma.
  const ctxWithRightmostOperator = {
    ...ctx,
    expressionRoot: rightmostOperator
  };
  const specialSuggestions = await (0, _dispatcher.dispatchOperators)(ctxWithRightmostOperator);
  if (specialSuggestions) {
    return specialSuggestions;
  }
  const getExprType = expression => (0, _expressions.getExpressionType)(expression, context === null || context === void 0 ? void 0 : context.columns);
  const {
    complete,
    reason
  } = isOperatorComplete(rightmostOperator, getExprType);
  if (complete) {
    return handleCompleteOperator(ctx, rightmostOperator, getExprType);
  }
  return handleIncompleteOperator(ctx, rightmostOperator, getExprType, reason);
}

/** Checks if an operator invocation is complete and correctly typed */
function isOperatorComplete(func, getExprType) {
  const fnDefinition = (0, _functions.getFunctionDefinition)(func.name);
  if (!fnDefinition) {
    return {
      complete: false
    };
  }
  const cleanedArgs = (0, _shared.removeFinalUnknownIdentiferArg)(func.args, getExprType);
  const argLengthCheck = fnDefinition.signatures.some(({
    minParams,
    params
  }) => {
    if (minParams && cleanedArgs.length >= minParams) {
      return true;
    }
    if (cleanedArgs.length === params.length) {
      return true;
    }
    return cleanedArgs.length >= params.filter(({
      optional
    }) => !optional).length;
  });
  if (!argLengthCheck) {
    return {
      complete: false,
      reason: 'tooFewArgs'
    };
  }
  const givenTypes = func.args.map(arg => getExprType(arg));
  const literalMask = func.args.map(arg => (0, _is.isLiteral)(Array.isArray(arg) ? arg[0] : arg));
  const hasCorrectTypes = !!(0, _expressions.getMatchingSignatures)(fnDefinition.signatures, givenTypes, literalMask, true).length;
  if (!hasCorrectTypes) {
    return {
      complete: false,
      reason: 'wrongTypes'
    };
  }
  return {
    complete: true
  };
}

/** Returns supported right-side types for binary operators matching the left-side type */
function getSupportedTypesForBinaryOperators(fnDef, previousType) {
  return fnDef ? fnDef.signatures.filter(({
    params
  }) => params.find(({
    name,
    type
  }) => name === 'left' && type === previousType)).map(({
    params
  }) => {
    var _params$;
    return (_params$ = params[1]) === null || _params$ === void 0 ? void 0 : _params$.type;
  }).filter(type => type !== undefined && !(0, _types.isArrayType)(type)) : [previousType];
}

/**
 * Handles suggestions for complete operator expressions (e.g., field > 0 |)
 * Suggests operators that can continue the expression
 */
function handleCompleteOperator(ctx, operator, getExprType) {
  const {
    innerText,
    location,
    options
  } = ctx;
  const operatorReturnType = getExprType(operator);
  const builder = new _suggestion_builder.SuggestionBuilder(ctx);
  const operatorDecision = (0, _should_suggest_operators.shouldSuggestOperators)({
    expressionType: operatorReturnType,
    functionParameterContext: options.functionParameterContext,
    ctx
  });
  if (operatorDecision.shouldSuggest) {
    let allowed = operatorDecision.allowedOperators;
    if (!allowed && operatorReturnType === 'boolean') {
      allowed = _all_operators.logicalOperators.filter(({
        locationsAvailable
      }) => locationsAvailable.includes(location)).map(({
        name
      }) => name);
    }
    builder.addOperators({
      leftParamType: operatorReturnType === 'unknown' || operatorReturnType === 'unsupported' ? 'any' : operatorReturnType,
      ignored: ['=', ':'],
      allowed
    });
  }

  // Add comma using decision engine for all operators in function context
  if (options.functionParameterContext) {
    const analyzer = _signature_analyzer.SignatureAnalyzer.from(options.functionParameterContext);
    if (analyzer) {
      const typeMatches = analyzer.typeMatches(operatorReturnType, false);
      builder.addCommaIfNeeded({
        position: 'after_complete',
        typeMatches,
        isLiteral: false,
        hasMoreParams: analyzer.hasMoreParams,
        isVariadic: analyzer.isVariadic,
        hasMoreMandatoryArgs: analyzer.getHasMoreMandatoryArgs(),
        functionSignatures: analyzer.getValidSignatures(),
        isCursorFollowedByComma: false
      });
    }
  }
  const suggestions = builder.build();
  return suggestions.map(suggestion => {
    const overlap = (0, _shared.getOverlapRange)(innerText, suggestion.text);
    return {
      ...suggestion,
      rangeToReplace: overlap
    };
  });
}

/**
 * Handles suggestions for incomplete operator expressions (e.g., field > |)
 * Suggests fields, literals, and functions for the right-hand operand
 */
async function handleIncompleteOperator(ctx, operator, getExprType, reason) {
  const {
    innerText,
    options
  } = ctx;
  const builder = new _suggestion_builder.SuggestionBuilder(ctx);
  const cleanedArgs = (0, _shared.removeFinalUnknownIdentiferArg)(operator.args, getExprType);
  const leftArgType = getExprType(operator.args[cleanedArgs.length - 1]);
  if (reason === 'tooFewArgs') {
    var _getFunctionDefinitio, _options$addSpaceAfte, _options$openSuggesti;
    const fnDef = (0, _functions.getFunctionDefinition)(operator.name);
    if (fnDef !== null && fnDef !== void 0 && fnDef.signatures.every(({
      params
    }) => params.some(({
      type
    }) => (0, _types.isArrayType)(type)))) {
      return [_complete_items.listCompleteItem];
    }

    // AND/OR special case: suggest broader field/function set instead of just boolean
    const isAndOrWithBooleanLeft = leftArgType === 'boolean' && (operator.name === 'and' || operator.name === 'or') && ((_getFunctionDefinitio = (0, _functions.getFunctionDefinition)(operator.name)) === null || _getFunctionDefinitio === void 0 ? void 0 : _getFunctionDefinitio.type) === _types.FunctionDefinitionTypes.OPERATOR;
    const typeToUse = isAndOrWithBooleanLeft ? ['any'] : getSupportedTypesForBinaryOperators(fnDef, leftArgType);
    const useValueType = Boolean(operator.subtype === 'binary-expression');
    await builder.addFields({
      types: typeToUse,
      values: useValueType,
      addSpaceAfterField: (_options$addSpaceAfte = options.addSpaceAfterOperator) !== null && _options$addSpaceAfte !== void 0 ? _options$addSpaceAfte : false,
      openSuggestions: (_options$openSuggesti = options.openSuggestions) !== null && _options$openSuggesti !== void 0 ? _options$openSuggesti : false,
      promoteToTop: true
    }).then(b => {
      var _options$openSuggesti2, _options$addSpaceAfte2, _options$openSuggesti3;
      return b.addLiterals({
        types: typeToUse,
        includeDateLiterals: true,
        includeCompatibleLiterals: false,
        addComma: false,
        advanceCursorAndOpenSuggestions: (_options$openSuggesti2 = options.openSuggestions) !== null && _options$openSuggesti2 !== void 0 ? _options$openSuggesti2 : false
      }).addFunctions({
        types: typeToUse,
        ignoredFunctions: [],
        addSpaceAfterFunction: (_options$addSpaceAfte2 = options.addSpaceAfterOperator) !== null && _options$addSpaceAfte2 !== void 0 ? _options$addSpaceAfte2 : false,
        openSuggestions: (_options$openSuggesti3 = options.openSuggestions) !== null && _options$openSuggesti3 !== void 0 ? _options$openSuggesti3 : false
      });
    });
  }
  if (reason === 'wrongTypes') {
    if (leftArgType && options.preferredExpressionType) {
      if (leftArgType !== options.preferredExpressionType && leftArgType !== 'unknown' && leftArgType !== 'unsupported') {
        builder.addOperators({
          leftParamType: leftArgType,
          returnTypes: [options.preferredExpressionType]
        });
      }
    }
  }
  return builder.build().map(suggestion => {
    const overlap = (0, _shared.getOverlapRange)(innerText, suggestion.text);
    return {
      ...suggestion,
      rangeToReplace: overlap
    };
  });
}

/**
 * Finds the deepest binary operator in the right branch of an expression tree.
 *
 * Uses structural right-first traversal instead of position-based detection
 * to avoid ANTLR error recovery issues where incomplete expressions get
 * positions that corrupt location.min values.
 */
function getRightmostOperatorInFunctionTree(fn) {
  const rightArg = fn.args[1];
  if (fn.subtype === 'binary-expression' && rightArg && !Array.isArray(rightArg) && rightArg.type === 'function') {
    return getRightmostOperatorInFunctionTree(rightArg);
  }
  return fn;
}