"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.postAttackDiscoveryGenerationsDismissRoute = void 0;
var _actions = require("@kbn/security-solution-features/actions");
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _elasticAssistantCommon = require("@kbn/elastic-assistant-common");
var _zodHelpers = require("@kbn/zod-helpers");
var _constants = require("../../../../../common/constants");
var _helpers = require("../../../helpers");
var _write_attack_discovery_event = require("./helpers/write_attack_discovery_event");
var _build_response = require("../../../../lib/build_response");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const postAttackDiscoveryGenerationsDismissRoute = router => {
  router.versioned.post({
    access: 'public',
    path: _elasticAssistantCommon.ATTACK_DISCOVERY_GENERATIONS_BY_ID_DISMISS,
    security: {
      authz: {
        requiredPrivileges: [_actions.ATTACK_DISCOVERY_API_ACTION_ALL]
      }
    }
  }).addVersion({
    version: _elasticAssistantCommon.API_VERSIONS.public.v1,
    validate: {
      request: {
        params: (0, _zodHelpers.buildRouteValidationWithZod)(_elasticAssistantCommon.PostAttackDiscoveryGenerationsDismissRequestParams)
      },
      response: {
        200: {
          body: {
            custom: (0, _zodHelpers.buildRouteValidationWithZod)(_elasticAssistantCommon.PostAttackDiscoveryGenerationsDismissResponse)
          }
        }
      }
    }
  }, async (context, request, response) => {
    const ctx = await context.resolve(['core', 'elasticAssistant', 'licensing']);
    const resp = (0, _build_response.buildResponse)(response);
    const assistantContext = await context.elasticAssistant;
    const logger = assistantContext.logger;

    // Perform license and authenticated user checks:
    const checkResponse = await (0, _helpers.performChecks)({
      context: ctx,
      request,
      response
    });
    if (!checkResponse.isSuccess) {
      return checkResponse.response;
    }
    try {
      const eventLogIndex = (await context.elasticAssistant).eventLogIndex;
      const eventLogger = (await context.elasticAssistant).eventLogger;
      const spaceId = (await context.elasticAssistant).getSpaceId();
      const dataClient = await assistantContext.getAttackDiscoveryDataClient();
      if (!dataClient) {
        return resp.error({
          body: `Attack discovery data client not initialized`,
          statusCode: 500
        });
      }
      const currentUser = await checkResponse.currentUser;
      const executionUuid = request.params.execution_uuid;
      const previousGeneration = await dataClient.getAttackDiscoveryGenerationById({
        authenticatedUser: currentUser,
        eventLogIndex,
        executionUuid,
        logger,
        spaceId
      });

      // event log details:
      const connectorId = previousGeneration.connector_id;
      await (0, _write_attack_discovery_event.writeAttackDiscoveryEvent)({
        action: _constants.ATTACK_DISCOVERY_EVENT_LOG_ACTION_GENERATION_DISMISSED,
        authenticatedUser: currentUser,
        connectorId,
        dataClient,
        eventLogger,
        eventLogIndex,
        executionUuid,
        loadingMessage: undefined,
        message: `Attack discovery generation ${executionUuid} for user ${currentUser.username} started`,
        spaceId
      });
      const latestGeneration = await dataClient.getAttackDiscoveryGenerationById({
        authenticatedUser: currentUser,
        eventLogIndex,
        executionUuid,
        logger,
        spaceId
      });
      return response.ok({
        body: {
          ...latestGeneration
        }
      });
    } catch (err) {
      logger.error(err);
      const error = (0, _securitysolutionEsUtils.transformError)(err);
      return resp.error({
        body: {
          success: false,
          error: error.message
        },
        statusCode: error.statusCode
      });
    }
  });
};
exports.postAttackDiscoveryGenerationsDismissRoute = postAttackDiscoveryGenerationsDismissRoute;