"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.postAttackDiscoveryBulkRoute = void 0;
var _actions = require("@kbn/security-solution-features/actions");
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _elasticAssistantCommon = require("@kbn/elastic-assistant-common");
var _zodHelpers = require("@kbn/zod-helpers");
var _helpers = require("../../../helpers");
var _build_response = require("../../../../lib/build_response");
var _index_privileges = require("../../helpers/index_privileges");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const postAttackDiscoveryBulkRoute = router => {
  router.versioned.post({
    access: 'public',
    path: _elasticAssistantCommon.ATTACK_DISCOVERY_BULK,
    security: {
      authz: {
        requiredPrivileges: [_actions.ATTACK_DISCOVERY_API_ACTION_ALL]
      }
    }
  }).addVersion({
    version: _elasticAssistantCommon.API_VERSIONS.public.v1,
    validate: {
      request: {
        body: (0, _zodHelpers.buildRouteValidationWithZod)(_elasticAssistantCommon.PostAttackDiscoveryBulkRequestBody)
      },
      response: {
        200: {
          body: {
            custom: (0, _zodHelpers.buildRouteValidationWithZod)(_elasticAssistantCommon.PostAttackDiscoveryBulkResponse)
          }
        }
      }
    }
  }, async (context, request, response) => {
    const ctx = await context.resolve(['core', 'elasticAssistant', 'licensing']);
    const resp = (0, _build_response.buildResponse)(response);
    const assistantContext = await context.elasticAssistant;
    const logger = assistantContext.logger;

    // Perform license and authenticated user checks:
    const checkResponse = await (0, _helpers.performChecks)({
      context: ctx,
      request,
      response
    });
    if (!checkResponse.isSuccess) {
      return checkResponse.response;
    }

    // Perform alerts access check
    const privilegesCheckResponse = await (0, _index_privileges.hasReadWriteAttackDiscoveryAlertsPrivileges)({
      context: ctx,
      response
    });
    if (!privilegesCheckResponse.isSuccess) {
      return privilegesCheckResponse.response;
    }
    try {
      var _request$body$update, _request$body$update2, _request$body$update3, _request$body$update$, _request$body$update4, _request$body$update$2, _request$body$update5;
      const currentUser = await checkResponse.currentUser;
      const dataClient = await assistantContext.getAttackDiscoveryDataClient();
      if (!dataClient) {
        return resp.error({
          body: `Attack discovery data client not initialized`,
          statusCode: 500
        });
      }
      const kibanaAlertWorkflowStatus = (_request$body$update = request.body.update) === null || _request$body$update === void 0 ? void 0 : _request$body$update.kibana_alert_workflow_status;
      const visibility = (_request$body$update2 = request.body.update) === null || _request$body$update2 === void 0 ? void 0 : _request$body$update2.visibility;
      const ids = (_request$body$update3 = request.body.update) === null || _request$body$update3 === void 0 ? void 0 : _request$body$update3.ids;
      const enableFieldRendering = (_request$body$update$ = (_request$body$update4 = request.body.update) === null || _request$body$update4 === void 0 ? void 0 : _request$body$update4.enable_field_rendering) !== null && _request$body$update$ !== void 0 ? _request$body$update$ : false; // public APIs default to rendering `james` instead of the `{{ user.name james }}` syntax, as a convenience to non-Kibana clients
      const withReplacements = (_request$body$update$2 = (_request$body$update5 = request.body.update) === null || _request$body$update5 === void 0 ? void 0 : _request$body$update5.with_replacements) !== null && _request$body$update$2 !== void 0 ? _request$body$update$2 : true; // public APIs default to applying replacements in responses as a convenience to non-Kibana clients

      if (ids == null || ids.length === 0) {
        return response.ok({
          body: {
            data: []
          }
        });
      }

      // get an Elasticsearch client for the authenticated user:
      const esClient = (await context.core).elasticsearch.client.asCurrentUser;
      const data = await dataClient.bulkUpdateAttackDiscoveryAlerts({
        authenticatedUser: currentUser,
        esClient,
        enableFieldRendering,
        ids,
        kibanaAlertWorkflowStatus,
        logger,
        visibility,
        withReplacements
      });
      return response.ok({
        body: {
          data
        }
      });
    } catch (err) {
      logger.error(err);
      const error = (0, _securitysolutionEsUtils.transformError)(err);
      return resp.error({
        body: {
          success: false,
          error: error.message
        },
        statusCode: error.statusCode
      });
    }
  });
};
exports.postAttackDiscoveryBulkRoute = postAttackDiscoveryBulkRoute;