"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.fetchDocuments = void 0;
var _i18n = require("@kbn/i18n");
var _rxjs = require("rxjs");
var _public = require("@kbn/data-plugin/public");
var _discoverUtils = require("@kbn/discover-utils");
var _public2 = require("@kbn/data-views-plugin/public");
var _get_allowed_sample_size = require("../../../utils/get_allowed_sample_size");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Requests the documents for Discover. This will return a promise that will resolve
 * with the documents.
 */
const fetchDocuments = (searchSource, {
  abortController,
  inspectorAdapters,
  searchSessionId,
  services,
  scopedProfilesManager,
  getCurrentTab
}) => {
  var _searchSource$getFiel;
  const {
    sampleSize
  } = getCurrentTab().appState;
  searchSource.setField('size', (0, _get_allowed_sample_size.getAllowedSampleSize)(sampleSize, services.uiSettings));
  searchSource.setField('trackTotalHits', false);
  searchSource.setField('highlightAll', true);
  searchSource.setField('version', true);
  if (((_searchSource$getFiel = searchSource.getField('index')) === null || _searchSource$getFiel === void 0 ? void 0 : _searchSource$getFiel.type) === _public2.DataViewType.ROLLUP) {
    // We treat that data view as "normal" even if it was a rollup data view,
    // since the rollup endpoint does not support querying individual documents, but we
    // can get them from the regular _search API that will be used if the data view
    // not a rollup data view.
    searchSource.setOverwriteDataViewType(undefined);
  }
  const dataView = searchSource.getField('index');
  const isFetchingMore = Boolean(searchSource.getField('searchAfter'));
  const executionContext = {
    description: isFetchingMore ? 'fetch more documents' : 'fetch documents'
  };
  const fetch$ = searchSource.fetch$({
    abortSignal: abortController.signal,
    sessionId: isFetchingMore ? undefined : searchSessionId,
    inspector: {
      adapter: inspectorAdapters.requests,
      title: isFetchingMore // TODO: show it as a separate request in Inspect flyout
      ? _i18n.i18n.translate('discover.inspectorRequestDataTitleMoreDocuments', {
        defaultMessage: 'More documents'
      }) : _i18n.i18n.translate('discover.inspectorRequestDataTitleDocuments', {
        defaultMessage: 'Documents'
      }),
      description: _i18n.i18n.translate('discover.inspectorRequestDescriptionDocument', {
        defaultMessage: 'This request queries Elasticsearch to fetch the documents.'
      })
    },
    executionContext,
    disableWarningToasts: true
  }).pipe((0, _rxjs.filter)(res => !(0, _public.isRunningResponse)(res)), (0, _rxjs.map)(res => {
    return (0, _discoverUtils.buildDataTableRecordList)({
      records: res.rawResponse.hits.hits,
      dataView,
      processRecord: record => scopedProfilesManager.resolveDocumentProfile({
        record
      })
    });
  }));
  return (0, _rxjs.lastValueFrom)(fetch$).then(records => {
    const adapter = inspectorAdapters.requests;
    const interceptedWarnings = [];
    if (adapter) {
      services.data.search.showWarnings(adapter, warning => {
        interceptedWarnings.push(warning);
        return true; // suppress the default behaviour
      });
    }
    return {
      records,
      interceptedWarnings
    };
  });
};
exports.fetchDocuments = fetchDocuments;