"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.fetchAll = fetchAll;
exports.fetchMoreDocuments = fetchMoreDocuments;
var _rxjs = require("rxjs");
var _esQuery = require("@kbn/es-query");
var _timerange = require("@kbn/timerange");
var _update_search_source = require("./update_search_source");
var _use_saved_search_messages = require("../hooks/use_saved_search_messages");
var _fetch_documents = require("./fetch_documents");
var _types = require("../../types");
var _fetch_esql = require("./fetch_esql");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * This function starts fetching all required queries in Discover. This will be the query to load the individual
 * documents as well as any other requests that might be required to load the main view.
 *
 * This method returns a promise, which will resolve (without a value), as soon as all queries that have been started
 * have been completed (failed or successfully).
 */
function fetchAll(params) {
  const {
    dataSubjects,
    reset = false,
    initialFetchStatus,
    services,
    scopedProfilesManager,
    scopedEbtManager,
    inspectorAdapters,
    savedSearch,
    abortController,
    getCurrentTab,
    onFetchRecordsComplete
  } = params;
  const {
    data,
    expressions
  } = services;
  try {
    const searchSource = savedSearch.searchSource.createChild();
    const dataView = searchSource.getField('index');
    const {
      query,
      sort
    } = getCurrentTab().appState;
    const isEsqlQuery = (0, _esQuery.isOfAggregateQueryType)(query);
    const currentTab = getCurrentTab();
    if (reset) {
      (0, _use_saved_search_messages.sendResetMsg)(dataSubjects, initialFetchStatus);
    }
    if (!isEsqlQuery) {
      // Update the base searchSource, base for all child fetches
      (0, _update_search_source.updateVolatileSearchSource)(searchSource, {
        dataView,
        services,
        sort: sort,
        inputTimeRange: currentTab.dataRequestParams.timeRangeAbsolute
      });
    }

    // Mark all subjects as loading
    (0, _use_saved_search_messages.sendLoadingMsg)(dataSubjects.main$);
    (0, _use_saved_search_messages.sendLoadingMsg)(dataSubjects.documents$, {
      query
    });
    (0, _use_saved_search_messages.sendLoadingMsg)(dataSubjects.totalHits$, {
      result: dataSubjects.totalHits$.getValue().result
    });

    // Start fetching all required requests
    const response = isEsqlQuery ? (0, _fetch_esql.fetchEsql)({
      query,
      dataView,
      abortSignal: abortController.signal,
      inspectorAdapters,
      data,
      expressions,
      scopedProfilesManager,
      timeRange: currentTab.dataRequestParams.timeRangeAbsolute,
      esqlVariables: currentTab.esqlVariables,
      searchSessionId: params.searchSessionId
    }) : (0, _fetch_documents.fetchDocuments)(searchSource, params);
    const fetchType = isEsqlQuery ? 'fetchTextBased' : 'fetchDocuments';
    const fetchAllRequestOnlyTracker = scopedEbtManager.trackPerformanceEvent('discoverFetchAllRequestsOnly');

    // Calculate query range in seconds
    const queryRangeSeconds = currentTab.dataRequestParams.timeRangeAbsolute ? (0, _timerange.getTimeDifferenceInSeconds)(currentTab.dataRequestParams.timeRangeAbsolute) : 0;

    // Handle results of the individual queries and forward the results to the corresponding dataSubjects
    response.then(({
      records,
      esqlQueryColumns,
      interceptedWarnings = [],
      esqlHeaderWarning
    }) => {
      fetchAllRequestOnlyTracker.reportEvent({
        meta: {
          fetchType
        },
        key1: 'query_range_secs',
        value1: queryRangeSeconds
      });
      if (isEsqlQuery) {
        const fetchStatus = interceptedWarnings.filter(({
          type
        }) => type === 'incomplete').length > 0 ? _types.FetchStatus.ERROR : _types.FetchStatus.COMPLETE;
        dataSubjects.totalHits$.next({
          fetchStatus,
          result: records.length
        });
      } else {
        const currentTotalHits = dataSubjects.totalHits$.getValue();
        // If the total hits (or chart) query is still loading, emit a partial
        // hit count that's at least our retrieved document count
        if (currentTotalHits.fetchStatus === _types.FetchStatus.LOADING && !currentTotalHits.result) {
          // trigger `partial` only for the first request (if no total hits value yet)
          dataSubjects.totalHits$.next({
            fetchStatus: _types.FetchStatus.PARTIAL,
            result: records.length
          });
        }
      }

      /**
       * Determine the appropriate fetch status
       *
       * The partial state for ES|QL mode is necessary to limit data table renders.
       * Depending on the type of query new columns can be added to AppState to ensure the data table
       * shows the updated columns. The partial state was introduced to prevent
       * too frequent state changes that cause the table to re-render too often, which can cause
       * race conditions, poor user experience, and potential test flakiness.
       *
       * For non-ES|QL queries, we always use COMPLETE status as they don't require this
       * special handling.
       */
      const fetchStatus = isEsqlQuery ? _types.FetchStatus.PARTIAL : _types.FetchStatus.COMPLETE;
      dataSubjects.documents$.next({
        fetchStatus,
        result: records,
        esqlQueryColumns,
        esqlHeaderWarning,
        interceptedWarnings,
        query
      });
      (0, _use_saved_search_messages.checkHitCount)(dataSubjects.main$, records.length);
    })
    // In the case that the request was aborted (e.g. a refresh), swallow the abort error
    .catch(e => {
      if (!abortController.signal.aborted) throw e;
    })
    // Only the document query should send its errors to main$, to cause the full Discover app
    // to get into an error state. The other queries will not cause all of Discover to error out
    // but their errors will be shown in-place (e.g. of the chart).
    .catch(e => {
      (0, _use_saved_search_messages.sendErrorMsg)(dataSubjects.documents$, e, {
        query
      });
      (0, _use_saved_search_messages.sendErrorMsg)(dataSubjects.main$, e);
    });

    // Return a promise that will resolve once all the requests have finished or failed, or no results are found
    return (0, _rxjs.firstValueFrom)((0, _rxjs.race)((0, _rxjs.combineLatest)([isComplete(dataSubjects.documents$).pipe((0, _rxjs.switchMap)(async () => onFetchRecordsComplete === null || onFetchRecordsComplete === void 0 ? void 0 : onFetchRecordsComplete())), isComplete(dataSubjects.totalHits$)]), noResultsFound(dataSubjects.main$))).then(() => {
      // Send a complete message to main$ once all queries are done and if main$
      // is not already in an ERROR state, e.g. because the document query has failed.
      // This will only complete main$, if it hasn't already been completed previously
      // by a query finding no results.
      if (dataSubjects.main$.getValue().fetchStatus !== _types.FetchStatus.ERROR) {
        (0, _use_saved_search_messages.sendCompleteMsg)(dataSubjects.main$);
      }
    });
  } catch (error) {
    (0, _use_saved_search_messages.sendErrorMsg)(dataSubjects.main$, error);
    // We also want to return a resolved promise in an error case, since it just indicates we're done with querying.
    return Promise.resolve();
  }
}
async function fetchMoreDocuments(params) {
  const {
    dataSubjects,
    services,
    savedSearch,
    getCurrentTab
  } = params;
  try {
    var _lastDocuments, _lastDocuments$raw;
    const searchSource = savedSearch.searchSource.createChild();
    const dataView = searchSource.getField('index');
    const {
      query,
      sort
    } = getCurrentTab().appState;
    const isEsqlQuery = (0, _esQuery.isOfAggregateQueryType)(query);
    if (isEsqlQuery) {
      // not supported yet
      return;
    }
    const lastDocuments = dataSubjects.documents$.getValue().result || [];
    const lastDocumentSort = (_lastDocuments = lastDocuments[lastDocuments.length - 1]) === null || _lastDocuments === void 0 ? void 0 : (_lastDocuments$raw = _lastDocuments.raw) === null || _lastDocuments$raw === void 0 ? void 0 : _lastDocuments$raw.sort;
    if (!lastDocumentSort) {
      return;
    }
    searchSource.setField('searchAfter', lastDocumentSort);

    // Mark as loading
    (0, _use_saved_search_messages.sendLoadingMoreMsg)(dataSubjects.documents$);

    // Update the searchSource
    (0, _update_search_source.updateVolatileSearchSource)(searchSource, {
      dataView,
      services,
      sort: sort
    });

    // Fetch more documents
    const {
      records,
      interceptedWarnings
    } = await (0, _fetch_documents.fetchDocuments)(searchSource, params);

    // Update the state and finish the loading state
    (0, _use_saved_search_messages.sendLoadingMoreFinishedMsg)(dataSubjects.documents$, {
      moreRecords: records,
      interceptedWarnings
    });
  } catch (error) {
    (0, _use_saved_search_messages.sendLoadingMoreFinishedMsg)(dataSubjects.documents$, {
      moreRecords: [],
      interceptedWarnings: undefined
    });
    (0, _use_saved_search_messages.sendErrorTo)(dataSubjects.main$)(error);
  }
}
const isComplete = subject => {
  return subject.pipe((0, _rxjs.filter)(({
    fetchStatus
  }) => [_types.FetchStatus.COMPLETE, _types.FetchStatus.ERROR].includes(fetchStatus)), (0, _rxjs.distinctUntilChanged)((a, b) => a.fetchStatus === b.fetchStatus));
};
const noResultsFound = subject => {
  return subject.pipe((0, _rxjs.filter)(({
    fetchStatus,
    foundDocuments
  }) => fetchStatus === _types.FetchStatus.COMPLETE && !foundDocuments), (0, _rxjs.distinctUntilChanged)((a, b) => a.fetchStatus === b.fetchStatus && a.foundDocuments === b.foundDocuments));
};