"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getDataStreamDetails = getDataStreamDetails;
var _boom = require("@hapi/boom");
var _constants = require("../../../../common/constants");
var _es_fields = require("../../../../common/es_fields");
var _services = require("../../../services");
var _utils = require("../../../utils");
var _queries = require("../../../utils/queries");
var _get_failed_docs = require("../failed_docs/get_failed_docs");
var _get_data_streams = require("../get_data_streams");
var _get_data_streams_metering_stats = require("../get_data_streams_metering_stats");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function getDataStreamDetails({
  esClient,
  dataStream,
  start,
  end,
  isServerless
}) {
  throwIfInvalidDataStreamParams(dataStream);

  // Query datastreams as the current user as the Kibana internal user may not have all the required permissions
  const esClientAsCurrentUser = esClient.asCurrentUser;
  const esClientAsSecondaryAuthUser = esClient.asSecondaryAuthUser;
  const dataStreamPrivileges = (await _services.datasetQualityPrivileges.getHasIndexPrivileges(esClientAsCurrentUser, [dataStream], ['monitor', _constants.FAILURE_STORE_PRIVILEGE, _constants.MANAGE_FAILURE_STORE_PRIVILEGE]))[dataStream];
  const esDataStream = dataStreamPrivileges.monitor ? (await (0, _get_data_streams.getDataStreams)({
    esClient: esClientAsCurrentUser,
    datasetQuery: dataStream
  })).dataStreams[0] : undefined;
  try {
    var _await$getFailedDocsP;
    const dataStreamSummaryStats = await getDataStreamSummaryStats(esClientAsCurrentUser, dataStream, start, end);
    const failedDocs = !dataStreamPrivileges[_constants.FAILURE_STORE_PRIVILEGE] ? undefined : (_await$getFailedDocsP = await (0, _get_failed_docs.getFailedDocsPaginated)({
      esClient: esClientAsCurrentUser,
      types: [],
      datasetQuery: dataStream,
      start,
      end
    })) === null || _await$getFailedDocsP === void 0 ? void 0 : _await$getFailedDocsP[0];
    const avgDocSizeInBytes = dataStreamPrivileges.monitor && dataStreamSummaryStats.docsCount > 0 ? isServerless ? await getMeteringAvgDocSizeInBytes(esClientAsSecondaryAuthUser, dataStream) : await getAvgDocSizeInBytes(esClientAsCurrentUser, dataStream) : 0;
    const sizeBytes = Math.ceil(avgDocSizeInBytes * dataStreamSummaryStats.docsCount);
    return {
      ...dataStreamSummaryStats,
      failedDocsCount: failedDocs === null || failedDocs === void 0 ? void 0 : failedDocs.count,
      sizeBytes,
      hasFailureStore: esDataStream === null || esDataStream === void 0 ? void 0 : esDataStream.hasFailureStore,
      lastActivity: esDataStream === null || esDataStream === void 0 ? void 0 : esDataStream.lastActivity,
      userPrivileges: {
        canMonitor: dataStreamPrivileges.monitor,
        canReadFailureStore: dataStreamPrivileges[_constants.FAILURE_STORE_PRIVILEGE],
        canManageFailureStore: dataStreamPrivileges[_constants.MANAGE_FAILURE_STORE_PRIVILEGE]
      },
      customRetentionPeriod: esDataStream === null || esDataStream === void 0 ? void 0 : esDataStream.customRetentionPeriod,
      defaultRetentionPeriod: esDataStream === null || esDataStream === void 0 ? void 0 : esDataStream.defaultRetentionPeriod
    };
  } catch (e) {
    var _e$body, _e$body$error;
    // Respond with empty object if data stream does not exist
    if (e.statusCode === 404 || ((_e$body = e.body) === null || _e$body === void 0 ? void 0 : (_e$body$error = _e$body.error) === null || _e$body$error === void 0 ? void 0 : _e$body$error.type) === 'index_closed_exception') {
      return {};
    }
    throw e;
  }
}
const MAX_HOSTS = _constants.MAX_HOSTS_METRIC_VALUE + 1; // Adding 1 so that we can show e.g. '50+'

// Gather service.name terms
const serviceNamesAgg = {
  ['service.name']: {
    terms: {
      field: 'service.name',
      size: MAX_HOSTS
    }
  }
};
const entityFields = ['host.name', 'container.id', 'kubernetes.pod.uid', 'cloud.instance.id', 'aws.s3.bucket.name', 'aws.rds.db_instance.arn', 'aws.sqs.queue.name'];
function isFieldAggregatable(fieldCapsResponse, fieldName) {
  const fieldCaps = fieldCapsResponse.fields[fieldName];
  if (!fieldCaps) {
    return false;
  }
  return Object.values(fieldCaps).every(caps => caps.aggregatable === true);
}
async function getDataStreamSummaryStats(esClient, dataStream, start, end) {
  var _response$aggregation, _response$aggregation2, _response$aggregation3, _response$aggregation4;
  const datasetQualityESClient = (0, _utils.createDatasetQualityESClient)(esClient);
  const fieldCapsResponse = await esClient.fieldCaps({
    index: dataStream,
    fields: ['*'],
    include_unmapped: false
  });
  const aggregatableFields = entityFields.filter(field => isFieldAggregatable(fieldCapsResponse, field));

  // Gather host terms like 'host', 'pod', 'container'
  const hostsAgg = aggregatableFields.reduce((acc, idField) => ({
    ...acc,
    [idField]: {
      terms: {
        field: idField,
        size: MAX_HOSTS
      }
    }
  }), {});
  const response = await datasetQualityESClient.search({
    index: dataStream,
    query: (0, _queries.rangeQuery)(start, end)[0],
    size: 0,
    aggs: {
      total_count: {
        value_count: {
          field: '_index'
        }
      },
      degraded_count: {
        filter: {
          exists: {
            field: _es_fields._IGNORED
          }
        }
      },
      ...serviceNamesAgg,
      ...hostsAgg
    }
  });
  const docsCount = Number((_response$aggregation = (_response$aggregation2 = response.aggregations) === null || _response$aggregation2 === void 0 ? void 0 : _response$aggregation2.total_count.value) !== null && _response$aggregation !== void 0 ? _response$aggregation : 0);
  const degradedDocsCount = Number((_response$aggregation3 = (_response$aggregation4 = response.aggregations) === null || _response$aggregation4 === void 0 ? void 0 : _response$aggregation4.degraded_count.doc_count) !== null && _response$aggregation3 !== void 0 ? _response$aggregation3 : 0);
  return {
    docsCount,
    degradedDocsCount,
    services: getTermsFromAgg(serviceNamesAgg, response.aggregations),
    hosts: getTermsFromAgg(hostsAgg, response.aggregations)
  };
}
async function getMeteringAvgDocSizeInBytes(esClient, index) {
  var _meteringStats$index$, _meteringStats$index$2;
  const meteringStats = await (0, _get_data_streams_metering_stats.getDataStreamsMeteringStats)({
    esClient,
    dataStreams: [index]
  });
  const docCount = (_meteringStats$index$ = meteringStats[index].totalDocs) !== null && _meteringStats$index$ !== void 0 ? _meteringStats$index$ : 0;
  const sizeInBytes = (_meteringStats$index$2 = meteringStats[index].sizeBytes) !== null && _meteringStats$index$2 !== void 0 ? _meteringStats$index$2 : 0;
  return docCount ? sizeInBytes / docCount : 0;
}
async function getAvgDocSizeInBytes(esClient, index) {
  var _indexStats$_all$tota, _indexStats$_all$tota2, _indexStats$_all$tota3, _indexStats$_all$tota4, _indexStats$_all$tota5, _indexStats$_all$tota6;
  const indexStats = await esClient.indices.stats({
    index,
    forbid_closed_indices: false
  });
  const docCount = (_indexStats$_all$tota = (_indexStats$_all$tota2 = indexStats._all.total) === null || _indexStats$_all$tota2 === void 0 ? void 0 : (_indexStats$_all$tota3 = _indexStats$_all$tota2.docs) === null || _indexStats$_all$tota3 === void 0 ? void 0 : _indexStats$_all$tota3.count) !== null && _indexStats$_all$tota !== void 0 ? _indexStats$_all$tota : 0;
  const sizeInBytes = (_indexStats$_all$tota4 = (_indexStats$_all$tota5 = indexStats._all.total) === null || _indexStats$_all$tota5 === void 0 ? void 0 : (_indexStats$_all$tota6 = _indexStats$_all$tota5.store) === null || _indexStats$_all$tota6 === void 0 ? void 0 : _indexStats$_all$tota6.size_in_bytes) !== null && _indexStats$_all$tota4 !== void 0 ? _indexStats$_all$tota4 : 0;
  return docCount ? sizeInBytes / docCount : 0;
}
function getTermsFromAgg(termAgg, aggregations) {
  if (!aggregations) {
    return {};
  }
  return Object.entries(termAgg).reduce((acc, [key, _value]) => {
    var _aggregations$key;
    const values = (_aggregations$key = aggregations[key]) === null || _aggregations$key === void 0 ? void 0 : _aggregations$key.buckets.map(bucket => bucket.key);
    return {
      ...acc,
      [key]: values
    };
  }, {});
}
function throwIfInvalidDataStreamParams(dataStream) {
  if (!(dataStream !== null && dataStream !== void 0 && dataStream.trim())) {
    throw (0, _boom.badRequest)(`Data Stream name cannot be empty. Received value "${dataStream}"`);
  }
}