"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createDataStream = createDataStream;
exports.createOrUpdateDataStream = createOrUpdateDataStream;
exports.updateDataStreams = updateDataStreams;
var _lodash = require("lodash");
var _indexAdapter = require("@kbn/index-adapter");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const updateTotalFieldLimitSetting = async ({
  logger,
  esClient,
  indexName,
  totalFieldsLimit
}) => {
  logger.debug(`Updating total field limit setting for ${indexName} data stream.`);
  try {
    const settings = {
      'index.mapping.total_fields.limit': totalFieldsLimit
    };
    await (0, _indexAdapter.retryTransientEsErrors)(() => esClient.indices.putSettings({
      index: indexName,
      settings
    }), {
      logger
    });
  } catch (err) {
    logger.error(`Failed to PUT index.mapping.total_fields.limit settings for ${indexName}: ${err.message}`);
    throw err;
  }
};

// This will update the mappings but *not* the settings. This
// is due to the fact settings can be classed as dynamic and static, and static
// updates will fail on an index that isn't closed. New settings *will* be applied as part
// of the ILM policy rollovers. More info: https://github.com/elastic/kibana/pull/113389#issuecomment-940152654
const updateMapping = async ({
  logger,
  esClient,
  indexName,
  writeIndexOnly
}) => {
  logger.debug(`Updating mappings for ${indexName} data stream.`);
  let simulatedIndexMapping;
  try {
    simulatedIndexMapping = await (0, _indexAdapter.retryTransientEsErrors)(() => esClient.indices.simulateIndexTemplate({
      name: indexName
    }), {
      logger
    });
  } catch (err) {
    logger.error(`Ignored PUT mappings for ${indexName}; error generating simulated mappings: ${err.message}`);
    return;
  }
  const simulatedMapping = (0, _lodash.get)(simulatedIndexMapping, ['template', 'mappings']);
  if (simulatedMapping == null) {
    logger.error(`Ignored PUT mappings for ${indexName}; simulated mappings were empty`);
    return;
  }
  try {
    await (0, _indexAdapter.retryTransientEsErrors)(() => esClient.indices.putMapping({
      index: indexName,
      ...simulatedMapping,
      write_index_only: writeIndexOnly
    }), {
      logger
    });
  } catch (err) {
    logger.error(`Failed to PUT mapping for ${indexName}: ${err.message}`);
    throw err;
  }
};
/**
 * Updates the data stream mapping and total field limit setting
 */
const updateDataStreamMappings = async ({
  logger,
  esClient,
  totalFieldsLimit,
  indexNames,
  writeIndexOnly
}) => {
  // Update total field limit setting of found indices
  // Other index setting changes are not updated at this time
  await Promise.all(indexNames.map(indexName => updateTotalFieldLimitSetting({
    logger,
    esClient,
    totalFieldsLimit,
    indexName
  })));
  // Update mappings of the found indices.
  await Promise.all(indexNames.map(indexName => updateMapping({
    logger,
    esClient,
    totalFieldsLimit,
    indexName,
    writeIndexOnly
  })));
};
async function createOrUpdateDataStream({
  logger,
  esClient,
  name,
  totalFieldsLimit,
  writeIndexOnly
}) {
  logger.info(`Creating data stream - ${name}`);

  // check if data stream exists
  let dataStreamExists = false;
  try {
    const response = await (0, _indexAdapter.retryTransientEsErrors)(() => esClient.indices.getDataStream({
      name,
      expand_wildcards: 'all'
    }), {
      logger
    });
    dataStreamExists = response.data_streams.length > 0;
  } catch (error) {
    if ((error === null || error === void 0 ? void 0 : error.statusCode) !== 404) {
      logger.error(`Error fetching data stream for ${name} - ${error.message}`);
      throw error;
    }
  }

  // if a data stream exists, update the underlying mapping
  if (dataStreamExists) {
    await updateDataStreamMappings({
      logger,
      esClient,
      indexNames: [name],
      totalFieldsLimit,
      writeIndexOnly
    });
  } else {
    try {
      await (0, _indexAdapter.retryTransientEsErrors)(() => esClient.indices.createDataStream({
        name
      }), {
        logger
      });
    } catch (error) {
      var _error$meta, _error$meta$body, _error$meta$body$erro;
      if ((error === null || error === void 0 ? void 0 : (_error$meta = error.meta) === null || _error$meta === void 0 ? void 0 : (_error$meta$body = _error$meta.body) === null || _error$meta$body === void 0 ? void 0 : (_error$meta$body$erro = _error$meta$body.error) === null || _error$meta$body$erro === void 0 ? void 0 : _error$meta$body$erro.type) !== 'resource_already_exists_exception') {
        logger.error(`Error creating data stream ${name} - ${error.message}`);
        throw error;
      }
    }
  }
}
async function createDataStream({
  logger,
  esClient,
  name
}) {
  logger.debug(`Checking data stream exists - ${name}`);

  // check if data stream exists
  let dataStreamExists = false;
  try {
    const response = await (0, _indexAdapter.retryTransientEsErrors)(() => esClient.indices.getDataStream({
      name,
      expand_wildcards: 'all'
    }), {
      logger
    });
    dataStreamExists = response.data_streams.length > 0;
  } catch (error) {
    if ((error === null || error === void 0 ? void 0 : error.statusCode) !== 404) {
      logger.error(`Error fetching data stream for ${name} - ${error.message}`);
      throw error;
    }
  }

  // return if data stream already created
  if (dataStreamExists) {
    return;
  }
  logger.info(`Installing data stream - ${name}`);
  try {
    await (0, _indexAdapter.retryTransientEsErrors)(() => esClient.indices.createDataStream({
      name
    }), {
      logger
    });
  } catch (error) {
    var _error$meta2, _error$meta2$body, _error$meta2$body$err;
    if ((error === null || error === void 0 ? void 0 : (_error$meta2 = error.meta) === null || _error$meta2 === void 0 ? void 0 : (_error$meta2$body = _error$meta2.body) === null || _error$meta2$body === void 0 ? void 0 : (_error$meta2$body$err = _error$meta2$body.error) === null || _error$meta2$body$err === void 0 ? void 0 : _error$meta2$body$err.type) !== 'resource_already_exists_exception') {
      logger.error(`Error creating data stream ${name} - ${error.message}`);
      throw error;
    }
  }
}
async function updateDataStreams({
  logger,
  esClient,
  name,
  totalFieldsLimit,
  writeIndexOnly
}) {
  logger.info(`Updating data streams - ${name}`);

  // check if data stream exists
  let dataStreams = [];
  try {
    const response = await (0, _indexAdapter.retryTransientEsErrors)(() => esClient.indices.getDataStream({
      name,
      expand_wildcards: 'all'
    }), {
      logger
    });
    dataStreams = response.data_streams;
  } catch (error) {
    if ((error === null || error === void 0 ? void 0 : error.statusCode) !== 404) {
      logger.error(`Error fetching data stream for ${name} - ${error.message}`);
      throw error;
    }
  }
  if (dataStreams.length > 0) {
    await updateDataStreamMappings({
      logger,
      esClient,
      totalFieldsLimit,
      indexNames: dataStreams.map(dataStream => dataStream.name),
      writeIndexOnly
    });
  }
}