"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SearchSessionsMgmtAPI = void 0;
var _i18n = require("@kbn/i18n");
var _moment = _interopRequireDefault(require("moment"));
var _rxjs = require("rxjs");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

class SearchSessionsMgmtAPI {
  constructor(sessionsClient, config, deps) {
    this.sessionsClient = sessionsClient;
    this.config = config;
    this.deps = deps;
  }
  async fetchTableData({
    appId
  } = {}) {
    const mgmtConfig = this.config.management;
    const refreshTimeout = _moment.default.duration(mgmtConfig.refreshTimeout);
    const fetch$ = (0, _rxjs.from)(this.sessionsClient.find({
      page: 1,
      perPage: mgmtConfig.maxSessions,
      sortField: 'created',
      sortOrder: 'desc'
    }));
    const timeout$ = (0, _rxjs.timer)(refreshTimeout.asMilliseconds()).pipe((0, _rxjs.tap)(() => {
      this.deps.notifications.toasts.addDanger(_i18n.i18n.translate('data.mgmt.searchSessions.api.backgroundSearchFetchTimeout', {
        defaultMessage: 'Fetching the Background Search info timed out after {timeout} seconds',
        values: {
          timeout: refreshTimeout.asSeconds()
        }
      }));
    }), (0, _rxjs.mapTo)(null));

    // fetch the search sessions before timeout triggers
    try {
      const result = await (0, _rxjs.race)(fetch$, timeout$).toPromise();
      if (result && result.saved_objects) {
        return {
          savedObjects: result.saved_objects.filter(object => {
            if (!appId) return true;
            return object.attributes.appId === appId;
          }),
          statuses: result.statuses
        };
      }
    } catch (err) {
      // eslint-disable-next-line no-console
      console.error(err);
      this.deps.notifications.toasts.addError(err, {
        title: _i18n.i18n.translate('data.mgmt.searchSessions.api.fetchError', {
          defaultMessage: 'Failed to refresh the page!'
        })
      });
    }
    return {
      savedObjects: [],
      statuses: {}
    };
  }
  getExtendByDuration() {
    return this.config.defaultExpiration;
  }

  // Delete and expire
  async sendDelete(id) {
    var _this$deps$usageColle;
    (_this$deps$usageColle = this.deps.usageCollector) === null || _this$deps$usageColle === void 0 ? void 0 : _this$deps$usageColle.trackSessionDeleted();
    try {
      await this.sessionsClient.delete(id);
      this.deps.notifications.toasts.addSuccess({
        title: _i18n.i18n.translate('data.mgmt.searchSessions.api.backgroundSearchDeleted', {
          defaultMessage: 'The background search was deleted.'
        })
      });
    } catch (err) {
      this.deps.notifications.toasts.addError(err, {
        title: _i18n.i18n.translate('data.mgmt.searchSessions.api.backgroundSearchDeletedError', {
          defaultMessage: 'Failed to delete the background search!'
        })
      });
    }
  }

  // Extend
  async sendExtend(id, expires) {
    var _this$deps$usageColle2;
    (_this$deps$usageColle2 = this.deps.usageCollector) === null || _this$deps$usageColle2 === void 0 ? void 0 : _this$deps$usageColle2.trackSessionExtended();
    try {
      await this.sessionsClient.extend(id, expires);
      this.deps.notifications.toasts.addSuccess({
        title: _i18n.i18n.translate('data.mgmt.searchSessions.api.backgroundSearchExtended', {
          defaultMessage: 'The background search was extended.'
        })
      });
    } catch (err) {
      this.deps.notifications.toasts.addError(err, {
        title: _i18n.i18n.translate('data.mgmt.searchSessions.api.backgroundSearchExtendError', {
          defaultMessage: 'Failed to extend the background search!'
        })
      });
    }
  }

  // Change the user-facing name of a search session
  async sendRename(id, newName) {
    try {
      await this.sessionsClient.rename(id, newName);
      this.deps.notifications.toasts.addSuccess({
        title: _i18n.i18n.translate('data.mgmt.searchSessions.api.backgroundSearchRename', {
          defaultMessage: 'The background search was renamed'
        })
      });
    } catch (err) {
      this.deps.notifications.toasts.addError(err, {
        title: _i18n.i18n.translate('data.mgmt.searchSessions.api.backgroundSearchRenameError', {
          defaultMessage: 'Failed to rename the background search'
        })
      });
    }
  }
}
exports.SearchSessionsMgmtAPI = SearchSessionsMgmtAPI;