"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.StorageService = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _server = require("@kbn/core/server");
var _constants = require("../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Service for managing encrypted Cloud Connect API key storage
 */
class StorageService {
  constructor(dependencies) {
    (0, _defineProperty2.default)(this, "encryptedSavedObjectsClient", void 0);
    (0, _defineProperty2.default)(this, "savedObjectsClient", void 0);
    (0, _defineProperty2.default)(this, "logger", void 0);
    this.encryptedSavedObjectsClient = dependencies.encryptedSavedObjectsClient;
    this.savedObjectsClient = dependencies.savedObjectsClient;
    this.logger = dependencies.logger;
  }

  /**
   * Saves (or updates) the Cloud Connect API key
   * @param apiKey - The API key to store (will be encrypted)
   * @param clusterId - The cluster ID associated with this API key
   */
  async saveApiKey(apiKey, clusterId) {
    try {
      this.logger.debug('Saving Cloud Connect API key');
      const now = new Date().toISOString();
      const apiKeyData = {
        apiKey,
        clusterId,
        createdAt: now,
        updatedAt: now
      };
      await this.savedObjectsClient.create(_constants.CLOUD_CONNECT_API_KEY_TYPE, apiKeyData, {
        id: _constants.CLOUD_CONNECT_API_KEY_ID,
        overwrite: true // Replace if exists
      });
      this.logger.debug(`Cloud Connect API key saved successfully for cluster: ${clusterId}`);
    } catch (error) {
      this.logger.error('Failed to save Cloud Connect API key', {
        error
      });
      throw error;
    }
  }

  /**
   * Retrieves and decrypts the stored Cloud Connect API key
   * @returns The stored API key data, or undefined if not found
   */
  async getApiKey() {
    try {
      this.logger.debug('Retrieving Cloud Connect API key');
      const result = await this.encryptedSavedObjectsClient.getDecryptedAsInternalUser(_constants.CLOUD_CONNECT_API_KEY_TYPE, _constants.CLOUD_CONNECT_API_KEY_ID);
      this.logger.debug('Cloud Connect API key retrieved successfully');
      return result.attributes;
    } catch (error) {
      if (_server.SavedObjectsErrorHelpers.isNotFoundError(error)) {
        this.logger.debug('Cloud Connect API key not found');
        return undefined;
      }
      this.logger.error('Failed to retrieve Cloud Connect API key', {
        error
      });
      throw error;
    }
  }

  /**
   * Deletes the stored Cloud Connect API key
   */
  async deleteApiKey() {
    try {
      this.logger.debug('Deleting Cloud Connect API key');
      await this.savedObjectsClient.delete(_constants.CLOUD_CONNECT_API_KEY_TYPE, _constants.CLOUD_CONNECT_API_KEY_ID);
      this.logger.debug('Cloud Connect API key deleted successfully');
    } catch (error) {
      if (_server.SavedObjectsErrorHelpers.isNotFoundError(error)) {
        this.logger.debug('Cloud Connect API key not found (already deleted or never existed)');
        return;
      }
      this.logger.error('Failed to delete Cloud Connect API key', {
        error
      });
      throw error;
    }
  }
}
exports.StorageService = StorageService;