"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useServiceActions = void 0;
var _react = require("react");
var _i18n = require("@kbn/i18n");
var _app_context = require("../../../app_context");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const useServiceActions = ({
  onServiceUpdate,
  services
}) => {
  const {
    notifications,
    telemetryService,
    apiService,
    hasActionsSavePrivilege,
    setHasAnyDefaultLLMConnectors
  } = (0, _app_context.useCloudConnectedAppContext)();

  // Tracks which service is currently being updated (for loading spinner)
  const [loadingService, setLoadingService] = (0, _react.useState)(null);

  // Tracks the service pending disable confirmation (for modal)
  const [disableModalService, setDisableModalService] = (0, _react.useState)(null);

  /**
   * Core function to enable or disable a service via API.
   * On success, optimistically updates the UI via onServiceUpdate callback.
   */
  const handleServiceUpdate = async (serviceKey, enabled) => {
    var _service$config;
    setLoadingService(serviceKey);
    const {
      data,
      error
    } = await apiService.updateServices({
      [serviceKey]: {
        enabled
      }
    });
    if (error) {
      notifications.toasts.addDanger({
        title: enabled ? _i18n.i18n.translate('xpack.cloudConnect.services.enable.errorTitle', {
          defaultMessage: 'Failed to enable service'
        }) : _i18n.i18n.translate('xpack.cloudConnect.services.disable.errorTitle', {
          defaultMessage: 'Failed to disable service'
        }),
        text: error.message
      });
      setLoadingService(null);
      return;
    }
    if (data !== null && data !== void 0 && data.warning) {
      notifications.toasts.addWarning({
        title: enabled ? _i18n.i18n.translate('xpack.cloudConnect.services.enable.warningTitle', {
          defaultMessage: 'Service enabled with warnings'
        }) : _i18n.i18n.translate('xpack.cloudConnect.services.disable.warningTitle', {
          defaultMessage: 'Service disabled with warnings'
        }),
        text: data.warning
      });
    } else {
      notifications.toasts.addSuccess({
        title: enabled ? _i18n.i18n.translate('xpack.cloudConnect.services.enable.successTitle', {
          defaultMessage: 'Service enabled successfully'
        }) : _i18n.i18n.translate('xpack.cloudConnect.services.disable.successTitle', {
          defaultMessage: 'Service disabled successfully'
        })
      });
    }

    // Track telemetry for service enable/disable
    const service = services[serviceKey];
    const telemetryProps = {
      service_type: serviceKey,
      region_id: service === null || service === void 0 ? void 0 : (_service$config = service.config) === null || _service$config === void 0 ? void 0 : _service$config.region_id
    };
    if (enabled) {
      telemetryService.trackServiceEnabled(telemetryProps);
    } else {
      telemetryService.trackServiceDisabled(telemetryProps);
    }
    setLoadingService(null);
    // Optimistically update the UI
    onServiceUpdate(serviceKey, enabled);

    // When EIS is enabled and user has actions.save privilege, we know default LLM connectors were created
    if (serviceKey === 'eis' && enabled && hasActionsSavePrivilege && setHasAnyDefaultLLMConnectors) {
      setHasAnyDefaultLLMConnectors(true);
    }
  };

  // Enables a service directly without confirmation
  const handleEnableService = serviceKey => handleServiceUpdate(serviceKey, true);

  // Disables the service currently pending in the modal, then closes the modal
  const handleDisableService = async () => {
    if (!disableModalService) return;
    await handleServiceUpdate(disableModalService.key, false);
    setDisableModalService(null);
  };

  // Opens the disable confirmation modal for a specific service
  const showDisableModal = (serviceKey, serviceName) => {
    setDisableModalService({
      key: serviceKey,
      name: serviceName
    });
  };

  // Closes the disable confirmation modal without taking action
  const closeDisableModal = () => {
    setDisableModalService(null);
  };
  const handleEnableServiceByUrl = (serviceKey, url) => {
    // Track telemetry for external link click to enable service
    telemetryService.trackLinkClicked({
      destination_type: 'service_enable_url',
      service_type: serviceKey
    });
    window.open(url, '_blank');
  };
  return {
    loadingService,
    disableModalService,
    handleEnableService,
    handleDisableService,
    showDisableModal,
    closeDisableModal,
    handleEnableServiceByUrl
  };
};
exports.useServiceActions = useServiceActions;