"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.validateObservableValue = exports.validateObservableTypeKeyExists = exports.validateDuplicatedObservablesInRequest = exports.validateDuplicatedObservableTypesInRequest = exports.validateDuplicatedKeysInRequest = void 0;
var _boom = _interopRequireDefault(require("@hapi/boom"));
var _validators = require("../../common/observables/validators");
var _constants = require("../../common/constants");
var _observable_types = require("./observable_types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Throws an error if the request has custom fields with duplicated keys.
 */
const validateDuplicatedKeysInRequest = ({
  requestFields = [],
  fieldName
}) => {
  const uniqueKeys = new Set();
  const duplicatedKeys = new Set();
  requestFields.forEach(item => {
    if (uniqueKeys.has(item.key)) {
      duplicatedKeys.add(item.key);
    } else {
      uniqueKeys.add(item.key);
    }
  });
  if (duplicatedKeys.size > 0) {
    throw _boom.default.badRequest(`Invalid duplicated ${fieldName} keys in request: ${Array.from(duplicatedKeys.values())}`);
  }
};

/**
 * Throws an error if the request has observable types with duplicated labels.
 */
exports.validateDuplicatedKeysInRequest = validateDuplicatedKeysInRequest;
const validateDuplicatedObservableTypesInRequest = ({
  requestFields = []
}) => {
  const extractLabelFromItem = item => item.label.toLowerCase();
  const extractKeyFromItem = item => item.key.toLowerCase();

  // NOTE: this prevents adding duplicates for the builtin types
  const builtinLabels = _constants.OBSERVABLE_TYPES_BUILTIN.map(extractLabelFromItem);
  const builtinKeys = _constants.OBSERVABLE_TYPES_BUILTIN.map(extractKeyFromItem);
  const uniqueLabels = new Set(builtinLabels);
  const uniqueKeys = new Set(builtinKeys);
  const duplicatedLabels = new Set();
  requestFields.forEach(item => {
    const observableTypeLabel = extractLabelFromItem(item);
    const observableTypeKey = extractKeyFromItem(item);
    if (uniqueKeys.has(observableTypeKey)) {
      duplicatedLabels.add(observableTypeLabel);
    } else {
      uniqueKeys.add(observableTypeKey);
    }
    if (uniqueLabels.has(observableTypeLabel)) {
      duplicatedLabels.add(observableTypeLabel);
    } else {
      uniqueLabels.add(observableTypeLabel);
    }
  });
  if (duplicatedLabels.size > 0) {
    throw _boom.default.badRequest(`Invalid duplicated observable types in request: ${Array.from(duplicatedLabels.values())}`);
  }
};

/**
 * Throws an error if the request has observable types with duplicated labels.
 */
exports.validateDuplicatedObservableTypesInRequest = validateDuplicatedObservableTypesInRequest;
const validateDuplicatedObservablesInRequest = ({
  requestFields = []
}) => {
  const stringifyItem = item => [item.typeKey, item.value].join();
  const uniqueObservables = new Set();
  const duplicatedObservables = new Set();
  requestFields.forEach(item => {
    if (uniqueObservables.has(stringifyItem(item))) {
      duplicatedObservables.add(stringifyItem(item));
    } else {
      uniqueObservables.add(stringifyItem(item));
    }
  });
  if (duplicatedObservables.size > 0) {
    throw _boom.default.badRequest(`Invalid duplicated observables in request.`);
  }
};

/**
 * Throws an error if observable type key is not valid
 */
exports.validateDuplicatedObservablesInRequest = validateDuplicatedObservablesInRequest;
const validateObservableTypeKeyExists = async (casesClient, {
  caseOwner,
  observableTypeKey
}) => {
  const observableTypesSet = await (0, _observable_types.getAvailableObservableTypesMap)(casesClient, caseOwner);
  if (!observableTypesSet.has(observableTypeKey)) {
    throw _boom.default.badRequest(`Invalid observable type, key does not exist: ${observableTypeKey}`);
  }
};
exports.validateObservableTypeKeyExists = validateObservableTypeKeyExists;
const validateObservableValue = (observableTypeKey, observableValue) => {
  const validator = (0, _validators.getValidatorForObservableType)(observableTypeKey);
  const validationError = validator(observableValue);
  if (validationError) {
    throw _boom.default.badRequest(`Observable value "${observableValue}" is not valid for selected observable type ${observableTypeKey}.`);
  }
};
exports.validateObservableValue = validateObservableValue;