"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getManualAlertIds = exports.filterCaseAttachmentsBySearchTerm = void 0;
var _domain = require("../../../../common/types/domain");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getManualAlertIds = comments => {
  const dedupeAlerts = comments.reduce((alertIds, comment) => {
    if (comment.type === _domain.AttachmentType.alert) {
      const ids = Array.isArray(comment.alertId) ? comment.alertId : [comment.alertId];
      ids.forEach(id => alertIds.add(id));
      return alertIds;
    }
    return alertIds;
  }, new Set());
  return Array.from(dedupeAlerts);
};
exports.getManualAlertIds = getManualAlertIds;
const isAlertAttachment = comment => {
  return comment.type === _domain.AttachmentType.alert;
};
const filterAlertCommentByIds = (comment, searchTerm) => {
  const ids = Array.isArray(comment.alertId) ? comment.alertId : [comment.alertId];
  const filteredIds = ids.filter(id => Boolean(id) && id.includes(searchTerm));
  if (filteredIds.length === 0) {
    return null;
  }
  return {
    ...comment,
    alertId: filteredIds
  };
};
const isEventAttachment = comment => {
  return comment.type === _domain.AttachmentType.event;
};
const filterEventCommentByIds = (comment, searchTerm) => {
  const ids = Array.isArray(comment.eventId) ? comment.eventId : [comment.eventId];
  const filteredIds = ids.filter(id => Boolean(id) && id.includes(searchTerm));
  if (filteredIds.length === 0) {
    return null;
  }
  return {
    ...comment,
    eventId: filteredIds
  };
};
const filterCaseAttachmentsBySearchTerm = (caseData, searchTerm) => {
  if (!searchTerm) {
    return caseData;
  }
  return {
    ...caseData,
    comments: caseData.comments.map(comment => {
      if (isAlertAttachment(comment)) {
        return filterAlertCommentByIds(comment, searchTerm);
      }
      if (isEventAttachment(comment)) {
        return filterEventCommentByIds(comment, searchTerm);
      }
      return comment;
    }).filter(comment => comment !== null)
  };
};
exports.filterCaseAttachmentsBySearchTerm = filterCaseAttachmentsBySearchTerm;